package ch.iterial.keycloak.plugins.directus;

import org.jboss.logging.Logger;
import org.keycloak.events.admin.AdminEvent;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.utils.KeycloakModelUtils;

import java.util.HashSet;
import java.util.Set;

import static ch.iterial.keycloak.plugins.directus.UserRoleDto.ATTRIBUTE_DIRECTUS_ID;

public class UserRoleService {
    private static final Logger LOGGER = Logger.getLogger(UserRoleService.class);

    private final DirectusConnectionConfig connectionConfig;
    private final KeycloakSessionFactory keycloakSessionFactory;

    public UserRoleService(final DirectusConnectionConfig connectionConfig,
                           final KeycloakSessionFactory keycloakSessionFactory) {
        this.connectionConfig = connectionConfig;
        this.keycloakSessionFactory = keycloakSessionFactory;
    }

    public UserRoleDto getUser(final AdminEvent event) {
        final String userPath = event.getResourcePath();
        final String[] userIdTokens = userPath.split("/");
        if (userIdTokens.length < 2 || !userIdTokens[0].equalsIgnoreCase("users")) {
            throw new IllegalArgumentException("Invalid user resource path: " + userPath);
        }
        final String userId = userIdTokens[1];

        return KeycloakModelUtils.runJobInTransactionWithResult(keycloakSessionFactory, session -> {
            final RealmModel userRealm = session.realms().getRealm(event.getRealmId());
            final UserModel user = session.users().getUserById(userRealm, userId);

            if (user == null) {
                return new UserRoleDto(userId, null, null, null, UserRoleDto.STATUS_ACTIVE, null, null);
            }

            final Set<RoleModel> userRoles = extractUserRoles(user);
            final String roleName = connectionConfig.role();
            final String providerName = connectionConfig.provider() == null ? UserRoleDto.PROVIDER_KEYCLOAK : connectionConfig.provider();

            return new UserRoleDto(
                    user.getId(),
                    user.getFirstName(),
                    user.getLastName(),
                    user.getEmail(),
                    UserRoleDto.STATUS_ACTIVE,
                    providerName,
                    userRoles.stream()
                            .filter(role -> roleName == null || role.getName().equalsIgnoreCase(roleName))
                            .findFirst()
                            .map(role -> role.getFirstAttribute(ATTRIBUTE_DIRECTUS_ID))
                            .orElse(null)
            );
        });
    }

    private static Set<RoleModel> extractUserRoles(final UserModel user) {
        final Set<RoleModel> userRoles = new HashSet<>();

        user.getRoleMappingsStream().filter(role -> !role.isComposite()).forEach(role -> {
            LOGGER.info(String.format("User '%s' has non-composite role '%s'", user.getEmail(), role.getName()));
            userRoles.add(role);
        });
        user.getGroupsStream().forEach(group -> {
            LOGGER.info(String.format("User '%s' has group '%s'", user.getEmail(), group.getName()));
            group.getRoleMappingsStream().filter(role -> !role.isComposite()).forEach(role -> {
                LOGGER.info(String.format("Group '%s' has non-composite role '%s'", group.getName(), role.getName()));
                userRoles.add(role);
            });
        });

        return userRoles;
    }
}
