package ch.iterial.keycloak.plugins.directus;

import org.keycloak.events.admin.AdminEvent;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.utils.KeycloakModelUtils;

import java.util.Set;

import static ch.iterial.keycloak.plugins.directus.UserRoleDto.ATTRIBUTE_DIRECTUS_ID;

public class UserRoleService {
    private final DirectusConnectionConfig connectionConfig;
    private final KeycloakSessionFactory keycloakSessionFactory;

    public UserRoleService(final DirectusConnectionConfig connectionConfig,
                           final KeycloakSessionFactory keycloakSessionFactory) {
        this.connectionConfig = connectionConfig;
        this.keycloakSessionFactory = keycloakSessionFactory;
    }

    public UserRoleDto getUser(final AdminEvent event) {
        final String userPath = event.getResourcePath();
        final String[] userIdTokens = userPath.split("/");
        if (userIdTokens.length < 2 || !userIdTokens[0].equalsIgnoreCase("users")) {
            throw new IllegalArgumentException("Invalid user resource path: " + userPath);
        }
        final String userId = userIdTokens[1];

        return KeycloakModelUtils.runJobInTransactionWithResult(keycloakSessionFactory, session -> {
            final RealmModel userRealm = session.realms().getRealm(event.getRealmId());
            final UserModel user = session.users().getUserById(userRealm, userId);

            if (user == null) {
                return new UserRoleDto(userId, null, null, null, UserRoleDto.STATUS_ACTIVE, null, null);
            }

            final Set<RoleModel> userRoles = UserRoleUtils.extractUserRoles(user);
            final String[] configRoleNames = connectionConfig.roles();
            final String configProviderName = connectionConfig.provider();

            return new UserRoleDto(
                    user.getId(),
                    user.getFirstName(),
                    user.getLastName(),
                    user.getEmail(),
                    UserRoleDto.STATUS_ACTIVE,
                    configProviderName,
                    userRoles.stream()
                            .filter(role -> UserRoleUtils.roleNameMatches(configRoleNames, role))
                            .findFirst()
                            .map(role -> role.getFirstAttribute(ATTRIBUTE_DIRECTUS_ID))
                            .orElse(null)
            );
        });
    }
}
