package ch.iterial.keycloak.plugins.directus;

import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultClientConnectionReuseStrategy;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeader;
import org.jboss.logging.Logger;
import org.keycloak.Config.Scope;
import org.keycloak.events.EventListenerProvider;
import org.keycloak.events.EventListenerProviderFactory;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.provider.ProviderConfigProperty;
import org.keycloak.provider.ProviderConfigurationBuilder;

import java.io.IOException;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.PROVIDER;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.REALM;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.ROLES;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.ROLE_$key_ID;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.SYNC_CREATE;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.SYNC_DELETE;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.SYNC_UPDATE;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.TOKEN;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.TTL;
import static ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.PropertyNames.URL;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.apache.http.HttpHeaders.AUTHORIZATION;
import static org.apache.http.HttpHeaders.CONTENT_TYPE;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;
import static org.keycloak.provider.ProviderConfigProperty.BOOLEAN_TYPE;
import static org.keycloak.provider.ProviderConfigProperty.LIST_TYPE;
import static org.keycloak.provider.ProviderConfigProperty.STRING_TYPE;
import static org.keycloak.util.TokenUtil.TOKEN_TYPE_BEARER;

public class DirectusEventListenerProviderFactory implements EventListenerProviderFactory {

    private static final Logger LOGGER = Logger.getLogger(DirectusEventListenerProviderFactory.class);
    private static final String PROVIDER_ID = "directus-integration";

    private PoolingHttpClientConnectionManager connectionManager = null;
    private CloseableHttpClient httpClient = null;
    private DirectusNotifier directusNotifier = null;
    private UserRoleService userRoleService = null;
    private DirectusConnectionConfig connectionConfig = null;

    @Override
    public EventListenerProvider create(final KeycloakSession keycloakSession) {
        LOGGER.debug("Creating event listener provider: " + keycloakSession.toString());
        return new DirectusEventListenerProvider(directusNotifier, userRoleService, connectionConfig, keycloakSession);
    }

    @Override
    public void init(final Scope scope) {
        LOGGER.info("Factory init with scope: " + scope.getPropertyNames().stream().collect(Collectors.joining(",", "[", "]")));

        connectionConfig = DirectusConnectionConfig.fromScope(scope);

        LOGGER.info(String.format("Loaded %s config: " + connectionConfig, connectionConfig.valid() ? "valid" : "invalid"));

        configureHttpClient(connectionConfig);

        directusNotifier = new DirectusNotifier(httpClient, connectionConfig);
    }

    @Override
    public void postInit(final KeycloakSessionFactory keycloakSessionFactory) {
        LOGGER.info("Post init factory");

        userRoleService = new UserRoleService(connectionConfig, keycloakSessionFactory);
    }

    @Override
    public void close() {
        LOGGER.info("Closing factory");
        try {
            if (httpClient != null) {
                httpClient.close();
            }
            if (connectionManager != null) {
                connectionManager.close();
            }
        } catch (final IOException e) {
            LOGGER.error("Error closing http client", e);
        }
    }

    @Override
    public String getId() {
        return PROVIDER_ID;
    }

    @Override
    public List<ProviderConfigProperty> getConfigMetadata() {
        return ProviderConfigurationBuilder.create()
                .property().name(URL.getKey()).label("Directus base URL").type(STRING_TYPE).required(true).add()
                .property().name(TOKEN.getKey()).label("Directus admin user token").type(STRING_TYPE).required(true).secret(true).add()
                .property().name(ROLES.getKey()).label("Keycloak role names").type(LIST_TYPE).required(true).add()
                .property().name(ROLE_$key_ID.getKey()).label("Directus role ID").type(STRING_TYPE).required(true).add()
                .property().name(PROVIDER.getKey()).label("Directus auth provider name").type(STRING_TYPE).required(true).add()
                .property().name(REALM.getKey()).label("Keycloak realm").type(STRING_TYPE).add()
                .property().name(TTL.getKey()).label("Request TTL in milliseconds").type(STRING_TYPE).add()
                .property().name(SYNC_CREATE.getKey()).label("Should sync user creation").type(BOOLEAN_TYPE).defaultValue(false).add()
                .property().name(SYNC_UPDATE.getKey()).label("Should sync user update").type(BOOLEAN_TYPE).defaultValue(false).add()
                .property().name(SYNC_DELETE.getKey()).label("Should sync user deletion").type(BOOLEAN_TYPE).defaultValue(false).add()
                .build();
    }

    private synchronized void configureHttpClient(final DirectusConnectionConfig connectionConfig) {
        connectionManager = new PoolingHttpClientConnectionManager(connectionConfig.ttl(), MILLISECONDS);
        httpClient = HttpClientBuilder.create()
                .setKeepAliveStrategy(DefaultConnectionKeepAliveStrategy.INSTANCE)
                .setConnectionReuseStrategy(DefaultClientConnectionReuseStrategy.INSTANCE)
                .setConnectionManager(connectionManager)
                .setDefaultHeaders(Set.of(
                        new BasicHeader(CONTENT_TYPE, APPLICATION_JSON.toString()),
                        new BasicHeader(AUTHORIZATION, TOKEN_TYPE_BEARER + " " + connectionConfig.token())
                ))
                .build();
    }
}
