package ch.iterial.keycloak.plugins.directus;

import org.keycloak.Config.Scope;

import java.util.Arrays;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import static java.util.stream.Collectors.joining;

public record DirectusConnectionConfig(String url,
                                       String token,
                                       String realm,
                                       RoleConfig[] roles,
                                       String provider,
                                       long ttl,
                                       SyncConfig sync,
                                       boolean valid) {

    private static final String PROVIDER_KEYCLOAK = "keycloak";
    private static final Predicate<String[]> ARRAY_NOT_EMPTY = array -> array != null && array.length > 0;

    @Override
    public String toString() {
        return new StringJoiner(", ", DirectusConnectionConfig.class.getSimpleName() + "[", "]")
                .add("url='" + url + "'")
                .add("token='" + (token == null ? "null" : "********") + "'")
                .add("realm='" + realm + "'")
                .add("roles='" + (roles == null ? "[]" : Arrays.stream(roles).map(RoleConfig::toString).collect(joining(",", "[", "]"))) + "'")
                .add("provider='" + provider + "'")
                .add("ttl=" + ttl)
                .add("sync=" + sync)
                .toString();
    }

    public record SyncConfig(boolean create, boolean update, boolean delete) {
        @Override
        public String toString() {
            return new StringJoiner(", ", SyncConfig.class.getSimpleName() + "[", "]")
                    .add("create=" + create)
                    .add("update=" + update)
                    .add("delete=" + delete)
                    .toString();
        }
    }

    public record RoleConfig(String name, String directusId) {
        @Override
        public String toString() {
            return new StringJoiner(", ", RoleConfig.class.getSimpleName() + "[", "]")
                    .add("name='" + name + "'")
                    .add("directusID='" + directusId + "'")
                    .toString();
        }
    }

    public static DirectusConnectionConfig fromScope(final Scope scope) {
        final AtomicBoolean validity = new AtomicBoolean(true);
        final String url = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.URL);
        final String token = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.TOKEN);
        final String realm = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.REALM);
        final String[] roles = PropertyReaderFactory.forStrings(scope, validity).getValue(PropertyNames.ROLES);
        final String provider = PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.PROVIDER);
        final Long ttl = PropertyReaderFactory.forLong(scope, validity).getValue(PropertyNames.TTL);
        final Boolean syncCreate = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_CREATE);
        final Boolean syncUpdate = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_UPDATE);
        final Boolean syncDelete = PropertyReaderFactory.forBoolean(scope, validity).getValue(PropertyNames.SYNC_DELETE);

        return new DirectusConnectionConfig(
                PropertyUtils.trimToNull(url),
                PropertyUtils.trimToNull(token),
                PropertyUtils.trimToNull(realm),
                Optional.of(roles)
                        .filter(ARRAY_NOT_EMPTY)
                        .map(roleNames -> Arrays.stream(roleNames)
                                .map(roleName -> new RoleConfig(roleName, PropertyReaderFactory.forString(scope, validity).getValue(PropertyNames.ROLE_$key_ID, roleName)))
                                .toArray(RoleConfig[]::new))
                        .orElse(new RoleConfig[]{}),
                Optional.ofNullable(PropertyUtils.trimToNull(provider))
                        .orElse(PROVIDER_KEYCLOAK),
                PropertyUtils.toFinite(ttl),
                new SyncConfig(
                        PropertyUtils.isTrue(syncCreate),
                        PropertyUtils.isTrue(syncUpdate),
                        PropertyUtils.isTrue(syncDelete)
                ),
                validity.get()
        );
    }

    public enum PropertyNames {
        URL(true),
        TOKEN(true),
        REALM(false),
        ROLES(false),
        ROLE_$key_ID(false),
        PROVIDER(false),
        TTL(false),
        SYNC_CREATE(false),
        SYNC_UPDATE(false),
        SYNC_DELETE(false);

        private final boolean required;

        PropertyNames(final boolean required) {
            this.required = required;
        }

        public boolean isRequired() {
            return required;
        }

        public String getKey(final String... values) {
            return underscoreToDash(getEnumFullName(values).toLowerCase(Locale.ENGLISH));
        }

        public String getEnvVariableName(final String... values) {
            return "KC_DIRECTUS_" + dashToUnderscore(getEnumFullName(values)).toUpperCase(Locale.ENGLISH);
        }

        private String getEnumFullName(final String... values) {
            final String name = name();
            if (name.contains("_$key_") && values.length > 0) {
                return name.replace("_$key_", "_" + values[0] + "_");
            } else {
                return name;
            }
        }

        private static String underscoreToDash(final String string) {
            return string == null ? null : string.trim().replaceAll("\\s+", "-").replace("_", "-");
        }

        private static String dashToUnderscore(final String string) {
            return string == null ? null : string.trim().replaceAll("\\s+", "_").replace("-", "_");
        }
    }

    private record PropertyReader<T>(BiFunction<String, T, T> reader,
                                     Function<String, T> converter,
                                     AtomicBoolean validity) {
        public T getValue(final PropertyNames property, final String... values) {
            final T value = reader.apply(property.getKey(values), converter.apply(getSystemPropertiesValue(property, values)));
            final boolean invalid = value == null && property.isRequired();

            validity.set(validity.get() && !invalid);

            return value;
        }

        private static String getSystemPropertiesValue(final PropertyNames propertyName, final String... values) {
            return PropertyUtils.trimToNull(System.getenv(propertyName.getEnvVariableName(values)));
        }
    }

    private static abstract class PropertyReaderFactory {
        private PropertyReaderFactory() {
            super();
        }

        static PropertyReader<String> forString(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::get, PropertyUtils::trimToNull, validity);
        }

        static PropertyReader<String[]> forStrings(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>((key, fallback) -> Optional.ofNullable(scope.getArray(key))
                    .filter(ARRAY_NOT_EMPTY)
                    .orElse(fallback), PropertyUtils::toEmptyArray, validity);
        }

        static PropertyReader<Long> forLong(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::getLong, value -> value == null ? null : Long.valueOf(value), validity);
        }

        static PropertyReader<Boolean> forBoolean(final Scope scope, final AtomicBoolean validity) {
            return new PropertyReader<>(scope::getBoolean, value -> Boolean.TRUE.toString().equalsIgnoreCase(PropertyUtils.trimToNull(value)), validity);
        }
    }

    private static abstract class PropertyUtils {
        private PropertyUtils() {
            super();
        }

        static long toFinite(final Long value) {
            return value == null ? -1L : value;
        }

        static boolean isTrue(final Boolean value) {
            return value != null && value;
        }

        static String trimToNull(final String string) {
            if (string == null) {
                return null;
            }
            final String trimmed = string.trim();
            if (trimmed.isEmpty()) {
                return null;
            }
            return trimmed;
        }

        static String[] toEmptyArray(final String array) {
            return array == null
                    ? new String[]{}
                    : Arrays.stream(array.split(","))
                    .map(PropertyUtils::trimToNull)
                    .filter(Objects::nonNull)
                    .toArray(String[]::new);
        }
    }

}
