package ch.iterial.keycloak.plugins.directus;

import org.jboss.logging.Logger;
import org.keycloak.events.Event;
import org.keycloak.events.EventListenerProvider;
import org.keycloak.events.EventListenerTransaction;
import org.keycloak.events.EventType;
import org.keycloak.events.admin.AdminEvent;
import org.keycloak.events.admin.OperationType;
import org.keycloak.events.admin.ResourceType;
import org.keycloak.models.KeycloakSession;

import java.io.IOException;

public class DirectusEventListenerProvider implements EventListenerProvider {

    private static final Logger LOGGER = Logger.getLogger(DirectusEventListenerProvider.class);

    private final DirectusNotifier directusNotifier;
    private final UserRoleService userRoleService;
    private final DirectusConnectionConfig connectionConfig;
    private final EventListenerTransaction eventTransaction;

    public DirectusEventListenerProvider(final DirectusNotifier directusNotifier,
                                         final UserRoleService userRoleService,
                                         final DirectusConnectionConfig connectionConfig,
                                         final KeycloakSession keycloakSession) {
        this.directusNotifier = directusNotifier;
        this.userRoleService = userRoleService;
        this.connectionConfig = connectionConfig;
        this.eventTransaction = new EventListenerTransaction(this::sendAdminEvent, this::sendEvent);

        if (connectionConfig.valid()) {
            keycloakSession.getTransactionManager().enlistAfterCompletion(eventTransaction);
        }

        LOGGER.debug("Created provider");
    }

    @Override
    public void onEvent(final Event event) {
        LOGGER.debug(String.format("Provider handling event '%s': %s", event.getType(), event.getUserId()));
        if (connectionConfig.valid()) {
            eventTransaction.addEvent(event);
        }
    }

    @Override
    public void onEvent(final AdminEvent event, final boolean includeRepresentation) {
        LOGGER.debug(String.format("Provider handling admin event with representation '%s': %s", includeRepresentation, event.getOperationType()));
        if (connectionConfig.valid()) {
            eventTransaction.addAdminEvent(event, includeRepresentation);
        }
    }

    @Override
    public void close() {
        LOGGER.debug("Closing provider");
    }

    private void sendEvent(final Event event) {
        LOGGER.debug(String.format("Provider sending event '%s': %s", event.getType(), event.getUserId()));

        if (realmFilterPasses(event) && resourceFilterPasses(event)) {
            try {
                final UserRoleDto dto = userRoleService.getUser(event);

                LOGGER.info(String.format("Notifying Directus about the user event: %s", dto));

                switch (event.getType()) {
                    case REGISTER:
                        createOnDirectus(dto);
                        break;
                    default:
                        LOGGER.warn("Unsupported event type: " + event.getType());
                }
            } catch (final IOException e) {
                LOGGER.error("Failed to notify Directus about the user event", e);
            }
        }
    }

    private void sendAdminEvent(final AdminEvent event, final boolean includeRepresentation) {
        LOGGER.debug(String.format(
                "Provider sending admin event with representation '%s'. Realm ID: %s. Operation: %s. Resource: %s.",
                includeRepresentation,
                event.getRealmName(),
                event.getOperationType(),
                event.getResourceType() + " " + event.getResourcePath()
        ));

        if (realmFilterPasses(event) && resourceFilterPasses(event)) {
            try {
                final UserRoleDto dto = userRoleService.getUser(event);

                LOGGER.info(String.format("Notifying Directus about the user admin event: %s", dto));

                switch (event.getOperationType()) {
                    case CREATE:
                        createOnDirectus(dto);
                        break;
                    case UPDATE:
                        updateOnDirectus(dto);
                        break;
                    case DELETE:
                    case ACTION:
                    default:
                        LOGGER.warn("Unsupported operation type: " + event.getOperationType());
                }
            } catch (final Exception e) {
                LOGGER.error("Failed to notify Directus about the user change", e);
            }
        }
    }

    private boolean realmFilterPasses(final AdminEvent event) {
        return connectionConfig.realm() == null || connectionConfig.realm().equalsIgnoreCase(event.getRealmName());
    }

    private boolean realmFilterPasses(final Event event) {
        return connectionConfig.realm() == null || connectionConfig.realm().equalsIgnoreCase(event.getRealmName());
    }

    private boolean resourceFilterPasses(final AdminEvent event) {
        return ResourceType.USER.equals(event.getResourceType())
                && (connectionConfig.sync() == null
                || (OperationType.CREATE.equals(event.getOperationType()) && connectionConfig.sync().create())
                || (OperationType.UPDATE.equals(event.getOperationType()) && connectionConfig.sync().update())
                || (OperationType.DELETE.equals(event.getOperationType()) && connectionConfig.sync().delete()));
    }

    private boolean resourceFilterPasses(final Event event) {
        return EventType.REGISTER.equals(event.getType())
                && (connectionConfig.sync() == null || connectionConfig.sync().create());
    }

    private void createOnDirectus(final UserRoleDto dto) throws IOException {
        final UserRoleDto existing = directusNotifier.getUserByEmail(dto.email());
        if (existing == null) {
            directusNotifier.notifyUserCreated(dto);
            LOGGER.info("Notified Directus about new user creation: " + dto.email());
        } else {
            LOGGER.warn("User already exists in Directus with email: " + dto.email());

            directusNotifier.notifyUserUpdated(existing.id(), dto);
            LOGGER.info("Notified Directus about the user creation with upsert");
        }
    }

    private void updateOnDirectus(final UserRoleDto dto) throws IOException {
        final UserRoleDto existing = directusNotifier.getUserByEmail(dto.email());
        if (existing == null) {
            LOGGER.warn("User not found in Directus with email: " + dto.email());

            directusNotifier.notifyUserCreated(dto);
            LOGGER.info("Notified Directus about the user modification with upsert");
        } else {
            directusNotifier.notifyUserUpdated(existing.id(), dto);
            LOGGER.info("Notified Directus about the user modification with update: " + dto.email());
        }
    }
}
