package ch.iterial.keycloak.plugins.directus;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.jboss.logging.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;

public class DirectusNotifier {

    private static final Logger LOGGER = Logger.getLogger(DirectusNotifier.class);

    private final CloseableHttpClient httpClient;
    private final DirectusConnectionConfig connectionConfig;
    private final ObjectMapper objectMapper;

    public DirectusNotifier(final CloseableHttpClient httpClient,
                            final DirectusConnectionConfig connectionConfig) {
        this.httpClient = httpClient;
        this.connectionConfig = connectionConfig;
        this.objectMapper = new ObjectMapper();
    }

    public void notifyUserCreated(final UserRoleDto data) throws IOException {
        final HttpPost request = new HttpPost();
        request.setURI(URI.create(connectionConfig.url() + "/users"));
        request.setEntity(new StringEntity(objectMapper.writeValueAsString(data), ContentType.APPLICATION_JSON));

        notify(request);
    }

    public void notifyUserUpdated(final String id, final UserRoleDto updated) throws IOException {
        final UserRoleDto patch = new UserRoleDto(
                null,
                updated.userId(),
                updated.firstName(),
                updated.lastName(),
                null,
                null,
                null,
                updated.roleId()
        );
        final HttpPatch request = new HttpPatch();
        request.setURI(URI.create(connectionConfig.url() + "/users/" + id));
        request.setEntity(new StringEntity(objectMapper.writeValueAsString(patch), ContentType.APPLICATION_JSON));

        notify(request);
    }

    private void notify(final HttpEntityEnclosingRequestBase request) throws IOException {
        httpClient.execute(request, response -> {
            final int responseCode = response.getStatusLine().getStatusCode();
            try (final InputStream stream = response.getEntity().getContent()) {
                final String responseBody = new String(stream.readAllBytes(), StandardCharsets.UTF_8);
                if (isUnsuccessful(responseCode)) {
                    throw new IOException(String.format("Unable to send %s request to Directus: %s", request.getMethod(), responseBody));
                } else {
                    LOGGER.info(String.format("Successfully sent %s request to Directus: %s", request.getMethod(), responseBody));
                }
            }
            return Void.class;
        });
    }

    public UserRoleDto getUserByEmail(final String email) throws IOException {
        final HttpGet request = new HttpGet();
        request.setURI(URI.create(connectionConfig.url() + "/users?filter[email][_eq]=" + email));

        return httpClient.execute(request, response -> {
            final int responseCode = response.getStatusLine().getStatusCode();
            try (final InputStream stream = response.getEntity().getContent()) {
                if (isUnsuccessful(responseCode)) {
                    final String error = new String(stream.readAllBytes(), StandardCharsets.UTF_8);
                    throw new IOException("Unable to create new user on Directus: " + error);
                } else {
                    final UserRoleListDto results = objectMapper.readValue(stream, new TypeReference<>() {
                    });
                    if (results != null && results.data() != null && results.data().length > 0) {
                        final UserRoleDto result = results.data()[0];
                        LOGGER.info("Successfully retrieved user from Directus: " + result);
                        return result;
                    }
                }
            }
            return null;
        });
    }

    private static boolean isUnsuccessful(final int responseCode) {
        return responseCode < 200 || responseCode > 299;
    }
}
