package ch.iterial.keycloak.plugins.directus;

import ch.iterial.keycloak.plugins.directus.DirectusConnectionConfig.RoleConfig;
import org.keycloak.events.Event;
import org.keycloak.events.admin.AdminEvent;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.utils.KeycloakModelUtils;

import java.util.Arrays;
import java.util.Set;

public class UserRoleService {
    private final DirectusConnectionConfig connectionConfig;
    private final KeycloakSessionFactory keycloakSessionFactory;

    public UserRoleService(final DirectusConnectionConfig connectionConfig,
                           final KeycloakSessionFactory keycloakSessionFactory) {
        this.connectionConfig = connectionConfig;
        this.keycloakSessionFactory = keycloakSessionFactory;
    }

    public UserRoleDto getUser(final AdminEvent event) {
        final String userId = getUserId(event);
        return getByUserIdAndRealmId(userId, event.getRealmId());
    }

    public UserRoleDto getUser(final Event event) {
        final String userId = event.getUserId();
        return getByUserIdAndRealmId(userId, event.getRealmId());
    }

    private UserRoleDto getByUserIdAndRealmId(final String userId, final String realmId) {
        return KeycloakModelUtils.runJobInTransactionWithResult(keycloakSessionFactory, session -> {
            final RealmModel userRealm = session.realms().getRealm(realmId);
            final UserModel user = session.users().getUserById(userRealm, userId);
            final String configProviderName = connectionConfig.provider();

            if (user == null) {
                return new UserRoleDto(null, userId, null, null, null, UserRoleDto.STATUS_ACTIVE, configProviderName, null);
            }

            final Set<RoleModel> userRoles = UserRoleUtils.extractUserRoles(user);
            final RoleConfig[] roleConfigs = connectionConfig.roles();

            return new UserRoleDto(
                    null,
                    user.getId(),
                    user.getFirstName(),
                    user.getLastName(),
                    user.getEmail(),
                    UserRoleDto.STATUS_ACTIVE,
                    configProviderName,
                    Arrays.stream(roleConfigs)
                            .filter(roleConfig -> UserRoleUtils.roleNameMatches(userRoles, roleConfig))
                            .findFirst()
                            .map(RoleConfig::directusId)
                            .orElse(null)
            );
        });
    }

    private String getUserId(final AdminEvent event) {
        if (!connectionConfig.valid()) {
            throw new IllegalStateException("Connection config is not valid");
        }
        final String userPath = event.getResourcePath();
        final String[] userIdTokens = userPath == null || userPath.trim().isEmpty()
                ? new String[]{}
                : userPath.split("/");
        if (userIdTokens.length < 2 || !userIdTokens[0].equalsIgnoreCase("users")) {
            throw new IllegalArgumentException("Invalid user resource path: " + userPath);
        }
        return userIdTokens[1];
    }
}
