package ch.marcus_schulte.hivedoc;

import java.io.File;
import java.io.FileFilter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import nu.xom.Attribute;
import nu.xom.Builder;
import nu.xom.Document;
import nu.xom.Element;
import nu.xom.Elements;
import nu.xom.Text;

import org.apache.maven.model.Resource;
import org.apache.maven.project.MavenProject;
import org.apache.maven.reporting.AbstractMavenReport;
import org.apache.maven.reporting.MavenReport;
import org.apache.maven.reporting.MavenReportException;
import org.codehaus.doxia.site.renderer.SiteRenderer;

/**
 * Goal which adds hivemodule-docs to a project's site
 * 
 * @goal hivedoc
 * 
 * @phase generate-sources
 */
public class HivedocMojo extends AbstractMavenReport implements MavenReport {
	private final static String NS_SERVICE = "svc";

	private final static String NS_CONFIG_PONT = "conf";

	private final static String NS_SCHEMA = "sch";

	private final static String NS_CONTRIB = "ctr";

	private final static String NS_IMPL = "impl";

	private final static Map elementNamesToNamespace = new HashMap() {
		{
			put("service-point", NS_SERVICE);
			put("configuration-point", NS_CONFIG_PONT);
			put("schema", NS_SCHEMA);
			put("contribution", NS_CONTRIB);
			put("implementation", NS_IMPL);
		}
	};

	/**
	 * Generates the site report
	 * 
	 * @component
	 */
	private SiteRenderer siteRenderer;

	/**
	 * Location of the site.
	 * 
	 * @parameter default-value="${project.reporting.outputDirectory}"
	 * @required
	 */
	private File outDir;

	/**
	 * 
	 * The name of the sub-directory to contain the generated hivedocs.
	 * 
	 * @parameter default-value="hivedocs"
	 * 
	 * @required
	 * 
	 */
	private String reportSubDir;

	private File hiveDocsOutputDir;

	/**
	 * List of standard-resources to search for hivemodule.xml's.
	 * 
	 * Can't be changed by the user directly. To include
	 * additional resources set the optional parameter 
	 * extraResources
	 * 
	 * @parameter default-value="${project.resources}"
	 * @required
	 * 
	 */
	private List resources;
	 /**
	     * List of resources to search for hivemodule.xml's in addition
	     * to the standard resources.
	     *
	     * @parameter
	     * @optional
	     */

	    private String[] extraResources;
	    
	    /**
	     * Subdirectory containing the javadocs. Must be changed if you
	     * configure something non-default in the javadoc-plugin
	     	* 
	      * @parameter default-value="apidocs"
	      * @required 
	      */
	private String javadocDir;

	protected void executeReport(Locale arg0) throws MavenReportException {
		getLog().info("HiveDoc Mojo executing ... ");

 		hiveDocsOutputDir = new File( outDir, reportSubDir );

 		hiveDocsOutputDir.mkdirs();
		getSink().body();

 		for (Iterator it = resources.iterator(); it.hasNext();) {
 			Resource resource = (Resource) it.next();
			getLog().debug( "Scanning resource "+ resource.getDirectory() 
								+", "+resource.getTargetPath() );

 			File rdir = new File(resource.getDirectory());
 			processHivemodulesIn( rdir );					
 		}	

 		if (extraResources != null) {
			for (int i = 0; i < extraResources.length; i++) {
				getLog().debug("Scanning resource " + extraResources[i]);
				File er = new File(extraResources[i]);
				processHivemodulesIn(er);
			}
		}

		
		getSink().body_();
 		getSink().close();

 	}


	private void processHivemodulesIn(File dir) {
		File[] hivemodules = dir.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.isFile()
						&& pathname.getName().equals("hivemodule.xml");
			}
		});
		if (hivemodules == null) {
			getLog().debug("No hivemodules found in " + dir);
			return;
		}
		for (int i = 0; i < hivemodules.length; i++) {
			getLog().info("Hivedoc found " + hivemodules[i]);

			try {
				Document hiveModule = new Builder().build(hivemodules[i]);

				writeOutputForModule(hiveModule.getRootElement());
			} catch (Exception e) {
				throw new RuntimeException(e);
			}

			// new File( hiveDocsOutputDir, "hivedoc.xml" ).createNewFile();

		}

		File[] subdirs = dir.listFiles(new FileFilter() {
			public boolean accept(File pathname) {
				return pathname.isDirectory();
			}
		});
		for (int i = 0; i < subdirs.length; i++) {
			processHivemodulesIn(subdirs[i]);
		}
	}

	private void writeOutputForModule(Element module) {
		getSink().sectionTitle1();
		getSink().text(
				"HiveModule " + module.getAttributeValue("id") + " version "
						+ module.getAttributeValue("version"));
		getSink().sectionTitle1_();
		getSink().section1();

		getSink().text("Package ");
		String pkg = module.getAttributeValue("package");
		getSink().link(linkToJavadoc("package", pkg));
		getSink().text(pkg);
		getSink().link_();

		getSink().paragraph();
		directTextChildren(module);
		getSink().paragraph_();

		Elements servicePoints = module.getChildElements("service-point");
		Elements configPoints = module.getChildElements("configuration-point");
		Elements schemas = module.getChildElements("schema");
		Elements contribs = module.getChildElements("contribution");
		Elements overrides = module.getChildElements("implementation");

		writeSummaries(new Elements[] { servicePoints, configPoints, schemas,
				contribs, overrides }, new String[] { "Service-Points",
				"Configuration Points", "Schemas", "Contributions",
				"Implementations" });

		for (int i = 0; i < servicePoints.size(); i++)
			writeServicePoint(servicePoints.get(i), pkg);

		for (int i = 0; i < configPoints.size(); i++)
			writeConfigPoint(configPoints.get(i));

		for (int i = 0; i < schemas.size(); i++)
			writeSchema(schemas.get(i));

		for (int i = 0; i < contribs.size(); i++)
			writeContribution(contribs.get(i));

		for (int i = 0; i < overrides.size(); i++)
			writeImplementation(overrides.get(i), pkg);

		getSink().section1_();
	}

	private void directTextChildren(Element elem) {
		for (int i = 0; i < elem.getChildCount(); i++) {
			if (elem.getChild(i) instanceof Text)
				getSink().text(elem.getChild(i).getValue());
		}
	}

	private void writeConfigPoint(Element cpEl) {
		String id = cpEl.getAttributeValue("id");
		writeSubsectionTitle(NS_CONFIG_PONT, "Configuration Point", id);
		getSink().section2();

		String occ = cpEl.getAttributeValue("occurs");
		if (occ == null) {
			occ = "unbounded";
		}

		getSink().paragraph();
		getSink().bold();
		getSink().text("Occurrence: ");
		getSink().bold_();
		getSink().text(occ);
		getSink().paragraph_();

		getSink().paragraph();
		directTextChildren(cpEl);
		getSink().paragraph_();

		String sid = cpEl.getAttributeValue("schema-id");
		if (sid != null) {
			getSink().text("Schema: ");
			linkTo(NS_SCHEMA, sid);
		} else {
			writeSchema(cpEl.getFirstChildElement("schema"));
		}
		getSink().section2_();
	}

	private void writeContribution(Element cEl) {
		String id = cEl.getAttributeValue("configuration-id");
		writeSubsectionTitle(NS_CONTRIB, "Contribution to ", id);
		getSink().section2();

		getSink().paragraph();
		directTextChildren(cEl);
		getSink().paragraph_();

		getSink().verbatim(true);
		getSink().text(cEl.toXML());
		getSink().verbatim_();

		getSink().section2_();
	}

	private void writeImplementation(Element implEl, String pkg) {
		String id = implEl.getAttributeValue("service-id");
		writeSubsectionTitle(NS_IMPL, "Implementation of ", id);
		getSink().section2();

		getSink().paragraph();
		directTextChildren(implEl);
		getSink().paragraph_();

		writeServiceCreation(implEl, pkg);

		getSink().section2_();
	}

	private void writeSchema(Element sEl) {
		String id = sEl.getAttributeValue("id");
		if (id != null) {
			writeSubsectionTitle(NS_SCHEMA, "Schema", id);
			getSink().section2();
		}

		getSink().paragraph();
		directTextChildren(sEl);
		getSink().paragraph_();

		Elements elems = sEl.getChildElements("element");
		writeSchemaElementDefinitions(elems);
		if (id != null)
			getSink().section2_();
	}

	private void writeSchemaElementDefinitions(Elements elems) {
		getSink().bold();
		getSink().text("Elements");
		getSink().bold_();
		getSink().lineBreak();
		getSink().definitionList();
		for (int i = 0; i < elems.size(); i++) {
			Element e = elems.get(i);
			getSink().definedTerm();
			getSink().text(e.getAttributeValue("name"));
			getSink().definedTerm_();

			getSink().definition();
			directTextChildren(e);
			writeElementAttributeTable(e.getChildElements("attribute"));
			Elements subElements = e.getChildElements("element");
			if (subElements.size() > 0)
				writeSchemaElementDefinitions(subElements);
			getSink().definition_();
		}
		getSink().definitionList_();
	}

	private void writeElementAttributeTable(Elements attribs) {

		getSink().table();
		getSink().tableCaption();
		getSink().text("Attributes");
		getSink().tableCaption_();
		getSink().tableRow();
		getSink().tableHeaderCell();
		getSink().text("Name");
		getSink().tableHeaderCell_();
		getSink().tableHeaderCell();
		getSink().text("Required");
		getSink().tableHeaderCell_();
		getSink().tableHeaderCell();
		getSink().text("Translator");
		getSink().tableHeaderCell_();
		getSink().tableHeaderCell();
		getSink().text("Description");
		getSink().tableHeaderCell_();
		getSink().tableRow_();
		for (int i = 0; i < attribs.size(); i++) {
			Element a = attribs.get(i);
			getSink().tableRow();
			cell(a.getAttributeValue("name"));
			cell(a.getAttributeValue("required"));
			cell(a.getAttributeValue("translator"));
			getSink().tableCell();
			directTextChildren(a);
			getSink().tableCell_();
			getSink().tableRow_();
		}
		getSink().table_();
	}

	private void writeServicePoint(Element spEl, String pkg) {
		String id = spEl.getAttributeValue("id");
		writeSubsectionTitle(NS_SERVICE, "Service-Point", id);

		getSink().section2();
		String intf = spEl.getAttributeValue("interface");
		if (intf == null)
			intf = id;

		getSink().table();
		getSink().tableRow();

		cell("Interface:");
		getSink().tableCell();
		getSink().link(linkToJavadoc(intf, pkg));
		getSink().text(intf);
		getSink().link_();
		getSink().tableCell_();
		getSink().tableRow_();

		Attribute ps = spEl.getAttribute("parameters-schema-id");
		if (ps != null) {
			getSink().tableRow();
			cell("Factory-Parameters");
			getSink().tableCell();
			linkTo(NS_SCHEMA, ps.getValue());
			getSink().text(
					" (" + spEl.getAttributeValue("parameters-occurs") + ")");

			getSink().tableCell_();
			getSink().tableRow_();
		}

		getSink().tableRow();
		cell("Creation:");
		getSink().tableCell();

		// getSink().verbatim(true);
		// String cr = spEl.getChildElements().get(0).toXML();
		// getSink().text(cr);
		// getSink().verbatim_();
		writeServiceCreation(spEl, pkg);

		getSink().tableCell_();
		getSink().tableRow_();

		getSink().table_();

		getSink().paragraph();
		directTextChildren(spEl);
		getSink().paragraph_();

		getSink().section2_();
	}

	private void writeServiceCreation(Element parent, String pkg) {
		writeServiceCreationViaFactory(parent, pkg);
		writeDirectServiceCreation(parent, pkg);

		Element interceptor = parent.getFirstChildElement("interceptor");
		if (interceptor != null) {
			getSink().text("Intercepted by ");
			linkTo(NS_SERVICE, interceptor.getAttributeValue("service-id"));
		}

	}

	private void writeDirectServiceCreation(Element parent, String pkg) {
		Element instance = parent.getFirstChildElement("create-instance");
		if (instance != null) {
			String model = instance.getAttributeValue("model");
			if (model == null)
				model = "singleton";
			getSink().table();

			getSink().tableRow();
			cell("Service-Model");
			cell(model);
			getSink().tableRow_();

			getSink().tableRow();
			cell("Implementation-Class");
			getSink().tableCell();
			String implClass = instance.getAttributeValue("class");
			getSink().link(linkToJavadoc(implClass, pkg));
			getSink().text(implClass);
			getSink().link_();
			getSink().tableCell_();
			getSink().tableRow_();

			getSink().table_();
		}
	}

	private void writeServiceCreationViaFactory(Element parent, String pkg) {
		Element invocation = parent.getFirstChildElement("invoke-factory");
		if (invocation != null) {
			String id = invocation.getAttributeValue("service-id");
			if (id == null)
				id = "hivemind.BuilderFactory";
			String model = invocation.getAttributeValue("model");
			if (model == null)
				model = "singleton";
			String implClass = "";
			String parameters = "";
			if (invocation.getChildElements().size() > 0)

				if (id.equals("hivemind.BuilderFactory")) {
					Element construct = invocation
							.getFirstChildElement("construct");
					implClass = construct.getAttributeValue("class");
					for (int i = 0; i < construct.getChildElements().size(); i++)
						parameters += construct.getChildElements().get(i)
								.toXML()
								+ "\n";
				} else
					parameters = invocation.getChildElements().get(0).toXML();

			getSink().table();

			getSink().tableRow();
			cell("Factory-Service");
			getSink().tableCell();
			linkTo(NS_SERVICE, id);
			getSink().tableCell_();
			getSink().tableRow_();

			getSink().tableRow();
			cell("Service-Model");
			cell(model);
			getSink().tableRow_();

			getSink().tableRow();
			cell("Implementation-Class");
			getSink().tableCell();
			getSink().link(linkToJavadoc(implClass, pkg));
			getSink().text(implClass);
			getSink().link_();
			getSink().tableCell_();
			getSink().tableRow_();

			if (parameters.length() > 0) {
				getSink().tableRow();
				cell("Parameters");
				getSink().tableCell();
				getSink().verbatim(false);
				getSink().text(parameters);
				getSink().verbatim_();
				getSink().tableCell_();
				getSink().tableRow_();
			}

			getSink().table_();
		}
	}

	private void writeSubsectionTitle(String namespace, String category,
			String id) {
		getSink().sectionTitle2();
		getSink().anchor(namespace + "." + id);
		getSink().text(category + " " + id);
		getSink().anchor_();
		getSink().sectionTitle2_();
	}

	private void linkTo(String namespace, String id) {
		getSink().link("#" + namespace + "." + id);
		getSink().text(id);
		getSink().link_();
	}

	private void writeSummaries(Elements[] els, String[] captions) {
		getSink().table();
		getSink().tableCaption();
		getSink().text("Module Summary");
		getSink().tableCaption_();
		getSink().tableRow();
		for (int i = 0; i < captions.length; i++) {
			getSink().tableHeaderCell();
			getSink().text(captions[i]);
			getSink().tableHeaderCell_();
		}
		getSink().tableRow_();

		boolean rowsLeft = true;
		int row = 0;
		while (rowsLeft) {
			rowsLeft = false;
			getSink().tableRow();
			for (int i = 0; i < els.length; i++) {
				getSink().tableCell();
				if (els[i].size() > row) {
					Element e = els[i].get(row);

					String id = e.getAttributeValue("id");
					if (id == null)
						id = e.getAttributeValue("service-id");
					if (id == null) {
						id = e.getAttributeValue("configuration-id");
					}

					linkTo((String)elementNamesToNamespace.get(e.getLocalName()), id);

					if (els[i].size() > row + 1)
						rowsLeft = true;
				}
				getSink().tableCell_();
			}
			getSink().tableRow_();
			row++;
		}
		getSink().table_();
	}

	private void cell(String s) {
		getSink().tableCell();
		getSink().text(s == null ? "" : s);
		getSink().tableCell_();
	}

	private String linkToJavadoc(String target, String pkg) {
		if (target.split("\\.").length < 3) // heuristics :)
			target = pkg + "." + target;
		return "../" + javadocDir + "/" + target.replace('.', '/') + ".html";
	}

	protected String getOutputDirectory() {
		return reportSubDir;
	}

	protected MavenProject getProject() {
		return null;
	}

	protected SiteRenderer getSiteRenderer() {
		return siteRenderer;
	}

	public String getDescription(Locale arg0) {
		return "Documentation from HiveMind descriptors";
	}

	public String getName(Locale arg0) {
		return "HiveDocs";
	}

	public String getOutputName() {
		return reportSubDir + "/index";
	}

}
