package ch.mfrey.maven.plugin.filerename;

import java.io.File;
import java.io.IOException;
import java.util.Collection;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.FileFileFilter;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

@Mojo(name = "rename", defaultPhase = LifecyclePhase.GENERATE_SOURCES)
public class RenamerMojo extends AbstractMojo {
    public static class Replace {
        @Parameter(required = true)
        private String from;

        @Parameter(required = true)
        private String to;

        public String getFrom() {
            return from;
        }

        public void setFrom(String from) {
            this.from = from;
        }

        public String getTo() {
            return to;
        }

        public void setTo(String to) {
            this.to = to;
        }

        @Override
        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("Replace [from=").append(from).append(", to=").append(to).append("]");
            return builder.toString();
        }
    }

    @Parameter(required = true)
    private Replace[] replaces;

    @Parameter(defaultValue = "false")
    private boolean showfiles;

    @Parameter(defaultValue = "${project.build.outputDirectory}")
    private String basedir;

    @Parameter(readonly = true)
    private MavenProject project;

    public void execute() throws MojoExecutionException, MojoFailureException {
        File workingDir = new File(getBasedir());
        if (!workingDir.isAbsolute()) {
            workingDir = new File(project.getBasedir(), getBasedir());
        }
        getLog().info("Directory: " + workingDir.getAbsolutePath());
        for (Replace rep : getReplaces()) {
            getLog().info(rep.toString());
        }
        Collection<File> files = FileUtils.listFiles(workingDir, FileFileFilter.FILE, DirectoryFileFilter.DIRECTORY);
        try {
            for (File file : files) {
                String newPath = getNewPath(workingDir, file);
                if (!file.getAbsolutePath().equals(newPath)) {
                    File newFile = new File(newPath);
                    if (showfiles) {
                        getLog().info(
                                System.lineSeparator() + " - " + file.getAbsolutePath() + System.lineSeparator()
                                        + " + " + newFile.getAbsolutePath());
                    }
                    FileUtils.moveFile(file, newFile);
                }
            }
        } catch (IOException ex) {
            throw new MojoExecutionException("Could not rename file", ex);
        }
        try {
            cleanupEmptyDirs(workingDir);
        } catch (IOException ex) {
            throw new MojoExecutionException("Could not cleanup empty directories", ex);
        }

    }

    private String getNewPath(File workingDir, File file) throws MojoExecutionException {
        String path = file.getAbsolutePath().substring(workingDir.getAbsolutePath().length());
        Replace[] replaces = getReplaces();
        for (int i = 0; i < replaces.length; i++) {
            if (replaces[i].getFrom() == null || replaces[i].getTo() == null) {
                throw new MojoExecutionException("From and To cannot be NULL: " + replaces[i]);
            } else if (!replaces[i].getFrom().equals(replaces[i].getTo())) {
                path = path.replace(replaces[i].getFrom(), replaces[i].getTo());
            }
        }
        return workingDir.getAbsolutePath() + path;
    }

    private void cleanupEmptyDirs(File directory) throws IOException {
        if (directory.isDirectory()) {
            for (File file : directory.listFiles()) {
                if (file.isDirectory()) {
                    cleanupEmptyDirs(file);
                }
            }
            if (directory.list().length == 0) {
                FileUtils.deleteDirectory(directory);
            }
        }
    }

    public String getBasedir() {
        return basedir;
    }

    public void setBasedir(String basedir) {
        this.basedir = basedir;
    }

    public Replace[] getReplaces() {
        return replaces;
    }

    public void setReplaces(Replace[] replaces) {
        this.replaces = replaces;
    }

}
