/*
 *  Copyright (c) 2022 Raffael Herzog
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to
 *  deal in the Software without restriction, including without limitation the
 *  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 *  sell copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 *  IN THE SOFTWARE.
 */

package ch.raffael.meldioc.library.http.server.undertow.routing;

import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action1;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action2;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action3;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action4;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action5;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action6;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action7;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action8;
import ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action9;

import ch.raffael.meldioc.library.http.server.undertow.util.HttpStatus;
import ch.raffael.meldioc.library.http.server.undertow.util.HttpStatusException;
import io.undertow.server.HttpServerExchange;
import io.undertow.util.AttachmentKey;

import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

// vagen Thu Nov 17 08:23:55 CET 2022

/**
 * A capture of some value from the request (path segment, query parameter).
 *
 * <p>The name if usually irrelevant and it's no problem to reuse the same
 * name multiple times. It's just to be able to insert something meaningful
 * in error messages.
 */
public abstract class Capture<T> {

  private final String name;

  Capture(String name) {
    this.name = name;
  }

  public static <T> Capture<T> of(String name, Supplier<? extends T> fun) {
    return ofExtractor(name, __ -> fun.get());
  }

  public static <T> Capture<T> of(String name, Function<? super HttpServerExchange, ? extends T> fun) {
    return ofExtractor(name, t -> requireNonNull(fun.apply(t), "fun.apply(t)"));
  }

  public static <T> Capture<T> ofExtractor(String name, Extractor<? extends T> extractor) {
    return new OfExtractor<>(name, extractor);
  }

  private static <T> T invokeExtractor(HttpServerExchange exchange, Extractor<? extends T> extractor)
      throws HttpStatusException {
    try {
      return extractor.get(exchange);
    } catch (HttpStatusException | RuntimeException | Error e) {
      throw e;
    } catch (Exception e) {
      // TODO FIXME (2021-01-26) improve exception handling
      throw new HttpStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e);
    }
  }

  public String name() {
    return name;
  }

  abstract T get(HttpServerExchange exchange) throws HttpStatusException;


  // map self + 0

  public <R> Capture<R> map(Action1<? super T, ? extends R> action) {
    return map("f(" + name() + ")",
        action);
  }

  public <R> Capture<R> map(String name, Action1<? super T, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x)));
  }


  // map self + 1

  public <R, T1> Capture<R> map(Capture<? extends T1> v1, Action2<? super T, ? super T1, ? extends R> action) {
    return map("f(" + name()+","+v1.name() + ")",
        v1, action);
  }

  public <R, T1> Capture<R> map(String name, Capture<? extends T1> v1, Action2<? super T, ? super T1, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x)));
  }


  // map self + 2

  public <R, T1, T2> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Action3<? super T, ? super T1, ? super T2, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name() + ")",
        v1, v2, action);
  }

  public <R, T1, T2> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Action3<? super T, ? super T1, ? super T2, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x)));
  }


  // map self + 3

  public <R, T1, T2, T3> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Action4<? super T, ? super T1, ? super T2, ? super T3, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name() + ")",
        v1, v2, v3, action);
  }

  public <R, T1, T2, T3> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Action4<? super T, ? super T1, ? super T2, ? super T3, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x)));
  }


  // map self + 4

  public <R, T1, T2, T3, T4> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Action5<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name()+","+v4.name() + ")",
        v1, v2, v3, v4, action);
  }

  public <R, T1, T2, T3, T4> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Action5<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x), v4.get(x)));
  }


  // map self + 5

  public <R, T1, T2, T3, T4, T5> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Action6<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name()+","+v4.name()+","+v5.name() + ")",
        v1, v2, v3, v4, v5, action);
  }

  public <R, T1, T2, T3, T4, T5> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Action6<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x), v4.get(x), v5.get(x)));
  }


  // map self + 6

  public <R, T1, T2, T3, T4, T5, T6> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Action7<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name()+","+v4.name()+","+v5.name()+","+v6.name() + ")",
        v1, v2, v3, v4, v5, v6, action);
  }

  public <R, T1, T2, T3, T4, T5, T6> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Action7<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x), v4.get(x), v5.get(x), v6.get(x)));
  }


  // map self + 7

  public <R, T1, T2, T3, T4, T5, T6, T7> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Capture<? extends T7> v7, Action8<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name()+","+v4.name()+","+v5.name()+","+v6.name()+","+v7.name() + ")",
        v1, v2, v3, v4, v5, v6, v7, action);
  }

  public <R, T1, T2, T3, T4, T5, T6, T7> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Capture<? extends T7> v7, Action8<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x), v4.get(x), v5.get(x), v6.get(x), v7.get(x)));
  }


  // map self + 8

  public <R, T1, T2, T3, T4, T5, T6, T7, T8> Capture<R> map(Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Capture<? extends T7> v7, Capture<? extends T8> v8, Action9<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? extends R> action) {
    return map("f(" + name()+","+v1.name()+","+v2.name()+","+v3.name()+","+v4.name()+","+v5.name()+","+v6.name()+","+v7.name()+","+v8.name() + ")",
        v1, v2, v3, v4, v5, v6, v7, v8, action);
  }

  public <R, T1, T2, T3, T4, T5, T6, T7, T8> Capture<R> map(String name, Capture<? extends T1> v1, Capture<? extends T2> v2, Capture<? extends T3> v3, Capture<? extends T4> v4, Capture<? extends T5> v5, Capture<? extends T6> v6, Capture<? extends T7> v7, Capture<? extends T8> v8, Action9<? super T, ? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? extends R> action) {
    return new Mapped<>(name, x ->
        action.perform(get(x), v1.get(x), v2.get(x), v3.get(x), v4.get(x), v5.get(x), v6.get(x), v7.get(x), v8.get(x)));
  }

  // inner classes

  public static final class Attachment<T> extends Capture<T> {
    private final AttachmentKey<String> key = AttachmentKey.create(String.class);
    private final Converter<? extends T> converter;

    Attachment(String name, Converter<? extends T> converter) {
      super(name);
      this.converter = converter;
    }

    T get(HttpServerExchange exchange) throws HttpStatusException {
      return Objects.requireNonNullElseGet(converter.convert(name(), exchange.getAttachment(key)),
          () -> {
            throw new IllegalStateException("No captured value for '" + name() + "'");
          });
    }

    void capture(HttpServerExchange exchange, String value) {
      exchange.putAttachment(key, value);
    }
  }

  static abstract class Cached<T> extends Capture<T> {
    private final AttachmentKey<T> key = AttachmentKey.create(Object.class);

    Cached(String name) {
      super(name);
    }

    @Override
    T get(HttpServerExchange exchange) throws HttpStatusException {
      var value = exchange.getAttachment(key);
      if (value == null) {
        value = requireNonNull(extract(exchange), "extract(exchange)");
        exchange.putAttachment(key, value);
      }
      return value;
    }

    abstract T extract(HttpServerExchange exchange) throws HttpStatusException;
  }

  static final class Mapped<T> extends Cached<T> {
    private final Extractor<? extends T> extractor;
    Mapped(String name, Extractor<? extends T> extractor) {
      super(name);
      this.extractor = extractor;
    }
    @Override
    T extract(HttpServerExchange exchange) throws HttpStatusException {
      return invokeExtractor(exchange, extractor);
    }
  }

  static final class OfExtractor<T> extends Cached<T> {
    private final Extractor<? extends T> extractor;
    OfExtractor(String name, Extractor<? extends T> extractor) {
      super(name);
      this.extractor = extractor;
    }
    @Override
    T extract(HttpServerExchange exchange) throws HttpStatusException {
      return invokeExtractor(exchange, extractor);
    }
  }

  @FunctionalInterface
  interface Extractor<T> {
    T get(HttpServerExchange exchange) throws Exception;
  }
}
