/*
 *  Copyright (c) 2022 Raffael Herzog
 *  
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to
 *  deal in the Software without restriction, including without limitation the
 *  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 *  sell copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *  
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *  
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 *  IN THE SOFTWARE.
 */

package ch.raffael.meldioc.library.http.server.undertow.routing;

import ch.raffael.meldioc.library.http.server.undertow.codec.EmptyBody;
import ch.raffael.meldioc.library.http.server.undertow.codec.HttpDecoder;
import ch.raffael.meldioc.library.http.server.undertow.codec.HttpEncoder;
import ch.raffael.meldioc.library.http.server.undertow.handler.EndpointHandler;
import ch.raffael.meldioc.library.http.server.undertow.handler.EndpointHandler.State;
import ch.raffael.meldioc.library.http.server.undertow.util.HttpMethod;
import ch.raffael.meldioc.library.http.server.undertow.util.HttpStatus;
import io.vavr.collection.Set;
import io.vavr.control.Option;

import java.util.Arrays;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;

import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action0;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action0Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action1;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action1Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action2;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action2Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action3;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action3Void;
import static io.vavr.control.Option.some;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action4;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action4Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action5;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action5Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action6;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action6Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action7;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action7Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action8;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action8Void;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action9;
import static ch.raffael.meldioc.library.http.server.undertow.routing.Actions.Action9Void;

// vagen Thu Nov 17 08:23:54 CET 2022
/**
 * TODO JavaDoc
 */
public class EndpointBuilder<B, T> {
  final DslTrace trace;
  final BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback;
  final Set<HttpMethod> methods;
  final BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init;

  EndpointBuilder(DslTrace trace,
                   BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback,
                   Set<HttpMethod> methods,
                   BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
    this.trace = trace;
    this.updateCallback = updateCallback;
    this.methods = methods;
    this.init = init;
  }

  EndpointBuilder(EndpointBuilder<?, ?> prev,
                   BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
    this(prev, prev.trace, prev.methods, init);
  }

  EndpointBuilder(EndpointBuilder<?, ?> prev, Set<HttpMethod> methods,
                   BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
    this(prev, prev.trace, methods, init);
  }

  EndpointBuilder(EndpointBuilder<?, ?> prev, DslTrace trace, Set<HttpMethod> methods,
                   BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
    this.updateCallback = prev.updateCallback;
    this.trace = trace;
    this.methods = methods;
    this.init = init;
    updateCallback.accept(prev, this);
  }

  EndpointHandler<B, T> handler(Frame frame) {
    return init.apply(frame, EndpointHandler.initial());
  }

  <BB, RR> BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<BB, RR>> addInit(
      BiFunction<Frame, EndpointHandler<B, T>, EndpointHandler<BB, RR>> init) {
    return (f, h) -> init.apply(f, this.init.apply(f, h));
  }

  <BB, RR> BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<BB, RR>> addInit(
      Function<EndpointHandler<B, T>, EndpointHandler<BB, RR>> init) {
    return (f, h) -> init.apply(this.init.apply(f, h));
  }

  EndpointBuilder<B, T> fork(
      DslTrace trace,
      BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback) {
    return new EndpointBuilder<>(trace, updateCallback, methods, this.init);
  }

  public static class Method extends Decoding {
    Method(DslTrace trace, BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback,
           Set<HttpMethod> methods) {
      super(trace, updateCallback, methods);
    }

    Method(EndpointBuilder<?, ?> prev,
           BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<EmptyBody, EmptyBody>> init) {
      super(prev, init);
    }

    Method(EndpointBuilder<?, ?> prev, Set<HttpMethod> methods,
           BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<EmptyBody, EmptyBody>> init) {
      super(prev, methods, init);
    }

    public Method method(HttpMethod... methods) {
      return new Method(this, this.methods.addAll(Arrays.asList(methods)), init);
    }

    public Method get() {
      return method(HttpMethod.GET);
    }

    public Method head() {
      return method(HttpMethod.HEAD);
    }

    public Method post() {
      return method(HttpMethod.POST);
    }

    public Method put() {
      return method(HttpMethod.PUT);
    }

    public Method patch() {
      return method(HttpMethod.PATCH);
    }

    public Method delete() {
      return method(HttpMethod.DELETE);
    }
  }

  public static class Decoding extends Processing<EmptyBody, EmptyBody> {

    Decoding(DslTrace trace, BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback,
             Set<HttpMethod> methods) {
      super(trace, updateCallback, methods, (f, p) -> p);
    }

    Decoding(EndpointBuilder<?, ?> prev,
             BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<EmptyBody, EmptyBody>> init) {
      super(prev, init);
    }

    Decoding(EndpointBuilder<?, ?> prev, Set<HttpMethod> methods,
             BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<EmptyBody, EmptyBody>> init) {
      super(prev, methods, init);
    }

    public <T> Processing<T, T> accept(Codecs.DecoderSupplier<? extends T> decoder) {
      return new Processing<>(this, addInit((f, h) -> h.decoder(decoder.decoder(f))));
    }

    public <T> Processing<T, T> accept(HttpDecoder<? extends T> decoder) {
      return new Processing<>(this, addInit(h -> h.decoder(decoder)));
    }

    public <T> Processing<T, T> accept(Class<T> type) {
      return new Processing<>(this, addInit((f, h) -> h.decoder(f.dec.object(type))));
    }
  }

  public static class Processing<B, T> extends Response<B, T> {

    Processing(DslTrace trace, BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback,
               Set<HttpMethod> methods, BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>,
        EndpointHandler<B, T>> init) {
      super(trace, updateCallback, methods, init);
    }

    Processing(EndpointBuilder<?, ?> prev,
               BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
      super(prev, init);
    }

    Processing(EndpointBuilder<?, ?> prev, Set<HttpMethod> methods,
               BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
      super(prev, methods, init);
    }

    private <U> Processing<B, U> map0(Action1<? super State<? extends T>, ? extends U> invoke) {
      return new Processing<>(this, addInit(h -> h.processor(s -> {
        if (s.isException()) return s.promoteException();
        return s.value(invoke.perform(s));
      })));
    }

    private Processing<B, T> status0(Action1<? super State<? extends T>, ? extends Option<? extends HttpStatus>> invoke) {
      return new Processing<>(this, addInit(h -> h.processor(s -> {
        if (s.isException()) return s.promoteException();
        return s.httpStatus(Option.narrow(invoke.perform(s)));
      })));
    }

    private <X extends Throwable> Processing<B, T> recover0(Class<X> excType, Action2<? super X, ? super State<? extends T>, ? extends T> invoke) {
      return new Processing<>(this, addInit(h -> h.processor(s -> {
        if (!(s.isException() && excType.isInstance(s.exception()))) return s;
        return s.recover(invoke.perform(excType.cast(s.exception()), s));
      })));
    }

    private <X extends Throwable> Processing<B, T> exceptionStatus0(Class<X> excType, Action2<? super X, ? super State<? extends T>, ? extends Option<? extends HttpStatus>> invoke) {
      return new Processing<>(this, addInit(h -> h.processor(s -> {
        if (!(s.isException() && excType.isInstance(s.exception()))) return s;
        return s.exceptionHttpStatus(Option.narrow(invoke.perform(excType.cast(s.exception()), s)));
      })));
    }

    // 0 map body
    public <U> Processing<B, U> map(Action1<? super T, ? extends U> action) {
      return map0(s ->
          action.perform(s.value()));
    }

    // 0 map
    public <U> Processing<B, U> map(Action0<? extends U> action) {
      return map0(s ->
          action.perform());
    }

    // 0 tap body
    public Processing<B, T> tap(Action1Void<? super T> action) {
      return map0(s -> {
        action.perform(s.value());
        return s.value();
      });
    }

    // 0 tap
    public Processing<B, T> tap(Action0Void action) {
      return map0(s -> {
        action.perform();
        return s.value();
      });
    }

    // 0 consume body
    public Processing<B, EmptyBody> consume(Action1Void<? super T> action) {
      return map0(s -> {
        action.perform(s.value());
        return EmptyBody.instance();
      });
    }

    // 0 consume
    public Processing<B, EmptyBody> consume(Action0Void action) {
      return map0(s -> {
        action.perform();
        return EmptyBody.instance();
      });
    }

    public Processing<B, T> status(HttpStatus status) {
      return status0(__ -> some(status));
    }

    // 0 status body
    public Processing<B, T> status(Action1<? super T, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value()));
    }

    // 0 status
    public Processing<B, T> status(Action0<? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform());
    }

    // 0 recover exception
    public <X extends Throwable> Processing<B, T> recover(Class<X> excType, Action1<? super X, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e));
    }

    // 0 recover
    public <X extends Throwable> Processing<B, T> recover(Class<X> excType, Action0<? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform());
    }

    public <X extends Throwable> Processing<B, T> exceptionStatus(Class<X> excType, HttpStatus status) {
      return exceptionStatus0(excType, (e, s) -> some(status));
    }

    // 0 exceptionStatus exception
    public <X extends Throwable> Processing<B, T> exceptionStatus(Class<X> excType, Action1<? super X, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e));
    }

    // 0 exceptionStatus
    public <X extends Throwable> Processing<B, T> exceptionStatus(Class<X> excType, Action0<? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform());
    }

    // 0 pipe body
    public <V> Pipe<V> pipe(Action1<? super T, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value()));
    }

    // 0 pipe
    public <V> Pipe<V> pipe(Action0<? extends V> action) {
      return new Pipe<>(s ->
          action.perform());
    }

    // 1 map body
    public <U, P1> Processing<B, U> map(Capture<? extends P1> p1, Action2<? super T, ? super P1, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange())));
    }

    // 1 map
    public <U, P1> Processing<B, U> map(Capture<? extends P1> p1, Action1<? super P1, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange())));
    }

    // 1 tap body
    public <P1> Processing<B, T> tap(Capture<? extends P1> p1, Action2Void<? super T, ? super P1> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()));
        return s.value();
      });
    }

    // 1 tap
    public <P1> Processing<B, T> tap(Capture<? extends P1> p1, Action1Void<? super P1> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()));
        return s.value();
      });
    }

    // 1 consume body
    public <P1> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Action2Void<? super T, ? super P1> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 1 consume
    public <P1> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Action1Void<? super P1> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 1 status body
    public <P1> Processing<B, T> status(Capture<? extends P1> p1, Action2<? super T, ? super P1, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange())));
    }

    // 1 status
    public <P1> Processing<B, T> status(Capture<? extends P1> p1, Action1<? super P1, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange())));
    }

    // 1 recover exception
    public <X extends Throwable, P1> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Action2<? super X, ? super P1, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange())));
    }

    // 1 recover
    public <X extends Throwable, P1> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Action1<? super P1, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange())));
    }

    // 1 exceptionStatus exception
    public <X extends Throwable, P1> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Action2<? super X, ? super P1, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange())));
    }

    // 1 exceptionStatus
    public <X extends Throwable, P1> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Action1<? super P1, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange())));
    }

    // 1 pipe body
    public <V, P1> Pipe<V> pipe(Capture<? extends P1> p1, Action2<? super T, ? super P1, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange())));
    }

    // 1 pipe
    public <V, P1> Pipe<V> pipe(Capture<? extends P1> p1, Action1<? super P1, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange())));
    }

    // 2 map body
    public <U, P1, P2> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super T, ? super P1, ? super P2, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 map
    public <U, P1, P2> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Action2<? super P1, ? super P2, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 tap body
    public <P1, P2> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3Void<? super T, ? super P1, ? super P2> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()));
        return s.value();
      });
    }

    // 2 tap
    public <P1, P2> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Action2Void<? super P1, ? super P2> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()));
        return s.value();
      });
    }

    // 2 consume body
    public <P1, P2> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3Void<? super T, ? super P1, ? super P2> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 2 consume
    public <P1, P2> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Action2Void<? super P1, ? super P2> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 2 status body
    public <P1, P2> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super T, ? super P1, ? super P2, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 status
    public <P1, P2> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Action2<? super P1, ? super P2, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 recover exception
    public <X extends Throwable, P1, P2> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super X, ? super P1, ? super P2, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 recover
    public <X extends Throwable, P1, P2> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Action2<? super P1, ? super P2, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 exceptionStatus exception
    public <X extends Throwable, P1, P2> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super X, ? super P1, ? super P2, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 exceptionStatus
    public <X extends Throwable, P1, P2> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Action2<? super P1, ? super P2, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 pipe body
    public <V, P1, P2> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super T, ? super P1, ? super P2, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 2 pipe
    public <V, P1, P2> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Action2<? super P1, ? super P2, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange())));
    }

    // 3 map body
    public <U, P1, P2, P3> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super T, ? super P1, ? super P2, ? super P3, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 map
    public <U, P1, P2, P3> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3<? super P1, ? super P2, ? super P3, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 tap body
    public <P1, P2, P3> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4Void<? super T, ? super P1, ? super P2, ? super P3> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
        return s.value();
      });
    }

    // 3 tap
    public <P1, P2, P3> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3Void<? super P1, ? super P2, ? super P3> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
        return s.value();
      });
    }

    // 3 consume body
    public <P1, P2, P3> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4Void<? super T, ? super P1, ? super P2, ? super P3> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 3 consume
    public <P1, P2, P3> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3Void<? super P1, ? super P2, ? super P3> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 3 status body
    public <P1, P2, P3> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super T, ? super P1, ? super P2, ? super P3, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 status
    public <P1, P2, P3> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3<? super P1, ? super P2, ? super P3, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 recover exception
    public <X extends Throwable, P1, P2, P3> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super X, ? super P1, ? super P2, ? super P3, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 recover
    public <X extends Throwable, P1, P2, P3> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3<? super P1, ? super P2, ? super P3, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super X, ? super P1, ? super P2, ? super P3, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 exceptionStatus
    public <X extends Throwable, P1, P2, P3> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3<? super P1, ? super P2, ? super P3, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 pipe body
    public <V, P1, P2, P3> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super T, ? super P1, ? super P2, ? super P3, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 3 pipe
    public <V, P1, P2, P3> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action3<? super P1, ? super P2, ? super P3, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
    }

    // 4 map body
    public <U, P1, P2, P3, P4> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 map
    public <U, P1, P2, P3, P4> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4<? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 tap body
    public <P1, P2, P3, P4> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
        return s.value();
      });
    }

    // 4 tap
    public <P1, P2, P3, P4> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4Void<? super P1, ? super P2, ? super P3, ? super P4> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
        return s.value();
      });
    }

    // 4 consume body
    public <P1, P2, P3, P4> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 4 consume
    public <P1, P2, P3, P4> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4Void<? super P1, ? super P2, ? super P3, ? super P4> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 4 status body
    public <P1, P2, P3, P4> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 status
    public <P1, P2, P3, P4> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4<? super P1, ? super P2, ? super P3, ? super P4, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 recover exception
    public <X extends Throwable, P1, P2, P3, P4> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 recover
    public <X extends Throwable, P1, P2, P3, P4> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4<? super P1, ? super P2, ? super P3, ? super P4, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3, P4> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4<? super P1, ? super P2, ? super P3, ? super P4, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 pipe body
    public <V, P1, P2, P3, P4> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 4 pipe
    public <V, P1, P2, P3, P4> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action4<? super P1, ? super P2, ? super P3, ? super P4, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
    }

    // 5 map body
    public <U, P1, P2, P3, P4, P5> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 map
    public <U, P1, P2, P3, P4, P5> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 tap body
    public <P1, P2, P3, P4, P5> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
        return s.value();
      });
    }

    // 5 tap
    public <P1, P2, P3, P4, P5> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
        return s.value();
      });
    }

    // 5 consume body
    public <P1, P2, P3, P4, P5> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 5 consume
    public <P1, P2, P3, P4, P5> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 5 status body
    public <P1, P2, P3, P4, P5> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 status
    public <P1, P2, P3, P4, P5> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 recover exception
    public <X extends Throwable, P1, P2, P3, P4, P5> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 recover
    public <X extends Throwable, P1, P2, P3, P4, P5> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3, P4, P5> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4, P5> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 pipe body
    public <V, P1, P2, P3, P4, P5> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 5 pipe
    public <V, P1, P2, P3, P4, P5> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action5<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
    }

    // 6 map body
    public <U, P1, P2, P3, P4, P5, P6> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 map
    public <U, P1, P2, P3, P4, P5, P6> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 tap body
    public <P1, P2, P3, P4, P5, P6> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
        return s.value();
      });
    }

    // 6 tap
    public <P1, P2, P3, P4, P5, P6> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
        return s.value();
      });
    }

    // 6 consume body
    public <P1, P2, P3, P4, P5, P6> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 6 consume
    public <P1, P2, P3, P4, P5, P6> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 6 status body
    public <P1, P2, P3, P4, P5, P6> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 status
    public <P1, P2, P3, P4, P5, P6> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 recover exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 recover
    public <X extends Throwable, P1, P2, P3, P4, P5, P6> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4, P5, P6> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 pipe body
    public <V, P1, P2, P3, P4, P5, P6> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 6 pipe
    public <V, P1, P2, P3, P4, P5, P6> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action6<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
    }

    // 7 map body
    public <U, P1, P2, P3, P4, P5, P6, P7> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 map
    public <U, P1, P2, P3, P4, P5, P6, P7> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 tap body
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
        return s.value();
      });
    }

    // 7 tap
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
        return s.value();
      });
    }

    // 7 consume body
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 7 consume
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 7 status body
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 status
    public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 recover exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 recover
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 pipe body
    public <V, P1, P2, P3, P4, P5, P6, P7> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 7 pipe
    public <V, P1, P2, P3, P4, P5, P6, P7> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action7<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
    }

    // 8 map body
    public <U, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends U> action) {
      return map0(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 map
    public <U, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 tap body
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
        return s.value();
      });
    }

    // 8 tap
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
        return s.value();
      });
    }

    // 8 consume body
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9Void<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
      return map0(s -> {
        action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 8 consume
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 8 status body
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 status
    public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 recover exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 recover
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 exceptionStatus exception
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super X, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(e, p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 pipe body
    public <V, P1, P2, P3, P4, P5, P6, P7, P8> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 8 pipe
    public <V, P1, P2, P3, P4, P5, P6, P7, P8> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
    }

    // 9 map
    public <U, P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9, ? extends U> action) {
      return map0(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange())));
    }

    // 9 tap
    public <P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange()));
        return s.value();
      });
    }

    // 9 consume
    public <P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9Void<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9> action) {
      return map0(s -> {
        action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange()));
        return EmptyBody.instance();
      });
    }

    // 9 status
    public <P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, T> status(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9, ? extends Option<? extends HttpStatus>> action) {
      return status0(s ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange())));
    }

    // 9 recover
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, T> recover(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9, ? extends T> action) {
      return recover0(excType,
          (e, s) -> action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange())));
    }

    // 9 exceptionStatus
    public <X extends Throwable, P1, P2, P3, P4, P5, P6, P7, P8, P9> Processing<B, T> exceptionStatus(Class<X> excType, Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9, ? extends Option<? extends HttpStatus>> action) {
      return exceptionStatus0(excType, (e, s) ->
            action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange())));
    }

    // 9 pipe
    public <V, P1, P2, P3, P4, P5, P6, P7, P8, P9> Pipe<V> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Capture<? extends P9> p9, Action9<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? super P9, ? extends V> action) {
      return new Pipe<>(s ->
          action.perform(p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()), p9.get(s.exchange())));
    }

    public final class Pipe<V> {
      private final Action1<State<? extends T>, ? extends V> pipeFun;

      Pipe(Action1<State<? extends T>, ? extends V> pipeFun) {
        this.pipeFun = pipeFun;
      }

      // 0 pipe map body
      public <U> Processing<B, U> map(Action2<? super V, ? super T, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value()));
      }

      // 0 map
      public <U> Processing<B, U> map(Action1<? super V, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s)));
      }

      // 0 pipe tap body
      public Processing<B, T> tap(Action2Void<? super V, ? super T> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value());
          return s.value();
        });
      }

      // 0 pipe tap
      public Processing<B, T> tap(Action1Void<? super V> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s));
          return s.value();
        });
      }

      // 0 pipe consume body
      public Processing<B, EmptyBody> consume(Action2Void<? super V, ? super T> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value());
          return EmptyBody.instance();
        });
      }

      // 0 pipe consume
      public Processing<B, EmptyBody> consume(Action1Void<? super V> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s));
          return EmptyBody.instance();
        });
      }

      // 0 pipe pipe body
      public <U> Pipe<U> pipe(Action2<? super V, ? super T, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value()));
      }

      // 0 pipe pipe
      public <U> Pipe<U> pipe(Action1<? super V, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s)));
      }

      // 1 pipe map body
      public <U, P1> Processing<B, U> map(Capture<? extends P1> p1, Action3<? super V, ? super T, ? super P1, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange())));
      }

      // 1 map
      public <U, P1> Processing<B, U> map(Capture<? extends P1> p1, Action2<? super V, ? super P1, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange())));
      }

      // 1 pipe tap body
      public <P1> Processing<B, T> tap(Capture<? extends P1> p1, Action3Void<? super V, ? super T, ? super P1> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()));
          return s.value();
        });
      }

      // 1 pipe tap
      public <P1> Processing<B, T> tap(Capture<? extends P1> p1, Action2Void<? super V, ? super P1> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()));
          return s.value();
        });
      }

      // 1 pipe consume body
      public <P1> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Action3Void<? super V, ? super T, ? super P1> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 1 pipe consume
      public <P1> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Action2Void<? super V, ? super P1> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 1 pipe pipe body
      public <U, P1> Pipe<U> pipe(Capture<? extends P1> p1, Action3<? super V, ? super T, ? super P1, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange())));
      }

      // 1 pipe pipe
      public <U, P1> Pipe<U> pipe(Capture<? extends P1> p1, Action2<? super V, ? super P1, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange())));
      }

      // 2 pipe map body
      public <U, P1, P2> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Action4<? super V, ? super T, ? super P1, ? super P2, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange())));
      }

      // 2 map
      public <U, P1, P2> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super V, ? super P1, ? super P2, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange())));
      }

      // 2 pipe tap body
      public <P1, P2> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Action4Void<? super V, ? super T, ? super P1, ? super P2> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()));
          return s.value();
        });
      }

      // 2 pipe tap
      public <P1, P2> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3Void<? super V, ? super P1, ? super P2> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()));
          return s.value();
        });
      }

      // 2 pipe consume body
      public <P1, P2> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Action4Void<? super V, ? super T, ? super P1, ? super P2> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 2 pipe consume
      public <P1, P2> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3Void<? super V, ? super P1, ? super P2> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 2 pipe pipe body
      public <U, P1, P2> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Action4<? super V, ? super T, ? super P1, ? super P2, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange())));
      }

      // 2 pipe pipe
      public <U, P1, P2> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Action3<? super V, ? super P1, ? super P2, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange())));
      }

      // 3 pipe map body
      public <U, P1, P2, P3> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action5<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
      }

      // 3 map
      public <U, P1, P2, P3> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super V, ? super P1, ? super P2, ? super P3, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
      }

      // 3 pipe tap body
      public <P1, P2, P3> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action5Void<? super V, ? super T, ? super P1, ? super P2, ? super P3> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
          return s.value();
        });
      }

      // 3 pipe tap
      public <P1, P2, P3> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4Void<? super V, ? super P1, ? super P2, ? super P3> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
          return s.value();
        });
      }

      // 3 pipe consume body
      public <P1, P2, P3> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action5Void<? super V, ? super T, ? super P1, ? super P2, ? super P3> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 3 pipe consume
      public <P1, P2, P3> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4Void<? super V, ? super P1, ? super P2, ? super P3> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 3 pipe pipe body
      public <U, P1, P2, P3> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action5<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
      }

      // 3 pipe pipe
      public <U, P1, P2, P3> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Action4<? super V, ? super P1, ? super P2, ? super P3, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange())));
      }

      // 4 pipe map body
      public <U, P1, P2, P3, P4> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action6<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
      }

      // 4 map
      public <U, P1, P2, P3, P4> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
      }

      // 4 pipe tap body
      public <P1, P2, P3, P4> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action6Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
          return s.value();
        });
      }

      // 4 pipe tap
      public <P1, P2, P3, P4> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
          return s.value();
        });
      }

      // 4 pipe consume body
      public <P1, P2, P3, P4> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action6Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 4 pipe consume
      public <P1, P2, P3, P4> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 4 pipe pipe body
      public <U, P1, P2, P3, P4> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action6<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
      }

      // 4 pipe pipe
      public <U, P1, P2, P3, P4> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Action5<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange())));
      }

      // 5 pipe map body
      public <U, P1, P2, P3, P4, P5> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action7<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
      }

      // 5 map
      public <U, P1, P2, P3, P4, P5> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
      }

      // 5 pipe tap body
      public <P1, P2, P3, P4, P5> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action7Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
          return s.value();
        });
      }

      // 5 pipe tap
      public <P1, P2, P3, P4, P5> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
          return s.value();
        });
      }

      // 5 pipe consume body
      public <P1, P2, P3, P4, P5> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action7Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 5 pipe consume
      public <P1, P2, P3, P4, P5> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 5 pipe pipe body
      public <U, P1, P2, P3, P4, P5> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action7<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
      }

      // 5 pipe pipe
      public <U, P1, P2, P3, P4, P5> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Action6<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange())));
      }

      // 6 pipe map body
      public <U, P1, P2, P3, P4, P5, P6> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action8<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
      }

      // 6 map
      public <U, P1, P2, P3, P4, P5, P6> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
      }

      // 6 pipe tap body
      public <P1, P2, P3, P4, P5, P6> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action8Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
          return s.value();
        });
      }

      // 6 pipe tap
      public <P1, P2, P3, P4, P5, P6> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
          return s.value();
        });
      }

      // 6 pipe consume body
      public <P1, P2, P3, P4, P5, P6> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action8Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 6 pipe consume
      public <P1, P2, P3, P4, P5, P6> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 6 pipe pipe body
      public <U, P1, P2, P3, P4, P5, P6> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action8<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
      }

      // 6 pipe pipe
      public <U, P1, P2, P3, P4, P5, P6> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Action7<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange())));
      }

      // 7 pipe map body
      public <U, P1, P2, P3, P4, P5, P6, P7> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action9<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
      }

      // 7 map
      public <U, P1, P2, P3, P4, P5, P6, P7> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
      }

      // 7 pipe tap body
      public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action9Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
          return s.value();
        });
      }

      // 7 pipe tap
      public <P1, P2, P3, P4, P5, P6, P7> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
          return s.value();
        });
      }

      // 7 pipe consume body
      public <P1, P2, P3, P4, P5, P6, P7> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action9Void<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 7 pipe consume
      public <P1, P2, P3, P4, P5, P6, P7> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 7 pipe pipe body
      public <U, P1, P2, P3, P4, P5, P6, P7> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action9<? super V, ? super T, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), s.value(), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
      }

      // 7 pipe pipe
      public <U, P1, P2, P3, P4, P5, P6, P7> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Action8<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange())));
      }


      // 8 map
      public <U, P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, U> map(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends U> action) {
        return map0(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
      }

      // 8 pipe tap
      public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, T> tap(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
          return s.value();
        });
      }

      // 8 pipe consume
      public <P1, P2, P3, P4, P5, P6, P7, P8> Processing<B, EmptyBody> consume(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9Void<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> action) {
        return map0(s -> {
          action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange()));
          return EmptyBody.instance();
        });
      }

      // 8 pipe pipe
      public <U, P1, P2, P3, P4, P5, P6, P7, P8> Pipe<U> pipe(Capture<? extends P1> p1, Capture<? extends P2> p2, Capture<? extends P3> p3, Capture<? extends P4> p4, Capture<? extends P5> p5, Capture<? extends P6> p6, Capture<? extends P7> p7, Capture<? extends P8> p8, Action9<? super V, ? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8, ? extends U> action) {
        return new Pipe<>(s ->
            action.perform(pipeFun.perform(s), p1.get(s.exchange()), p2.get(s.exchange()), p3.get(s.exchange()), p4.get(s.exchange()), p5.get(s.exchange()), p6.get(s.exchange()), p7.get(s.exchange()), p8.get(s.exchange())));
      }
    }
  }

  public static class Response<B, T> extends EndpointBuilder<B, T> {

    Response(DslTrace trace, BiConsumer<EndpointBuilder<?, ?>, EndpointBuilder<?, ?>> updateCallback,
             Set<HttpMethod> methods, BiFunction<Frame,
        EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
      super(trace, updateCallback, methods, init);
    }

    Response(EndpointBuilder<?, ?> prev,
             BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
      super(prev, init);
    }

    Response(EndpointBuilder<?, ?> prev, Set<HttpMethod> methods,
             BiFunction<Frame, EndpointHandler<EmptyBody, EmptyBody>, EndpointHandler<B, T>> init) {
      super(prev, methods, init);
    }

    public EndpointBuilder<B, T> respond() {
      return respond(HttpStatus.OK);
    }

    public EndpointBuilder<B, T> respond(HttpStatus status) {
      return respond(status, Object.class);
    }

    public EndpointBuilder<B, T> respond(Class<? super T> type) {
      return respond(HttpStatus.OK, type);
    }

    public EndpointBuilder<B, T> respond(HttpStatus status, Class<? super T> type) {
      return new EndpointBuilder<>(this, addInit((f, h) -> h.defaultStatus(status).encoder(f.enc.object(type))));
    }

    public EndpointBuilder<B, T> respond(Codecs.EncoderSupplier<? super T> encoder) {
      return respond(HttpStatus.OK, encoder);
    }

    public EndpointBuilder<B, T> respond(HttpStatus status, Codecs.EncoderSupplier<? super T> encoder) {
      return new EndpointBuilder<>(this, addInit((f, h) -> h.defaultStatus(status).encoder(encoder.encoder(f))));
    }

    public EndpointBuilder<B, T> respond(HttpEncoder<? super T> encoder) {
      return respond(HttpStatus.OK, encoder);
    }

    public EndpointBuilder<B, T> respond(HttpStatus status, HttpEncoder<? super T> encoder) {
      return new EndpointBuilder<>(this, addInit(h -> h.defaultStatus(status).encoder(encoder)));
    }

    public EndpointBuilder<B, EmptyBody> respondEmpty() {
      return respondEmpty(HttpStatus.OK);
    }

    public EndpointBuilder<B, EmptyBody> respondEmpty(HttpStatus status) {
      return new EndpointBuilder<>(this,
          this.addInit(h ->
              h.processor(s -> s.map(__ -> EmptyBody.instance()))
                  .defaultStatus(status)
                  .encoder(EmptyBody.encoder())));
    }
  }
}
