/*
 *  Copyright (c) 2022 Raffael Herzog
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to
 *  deal in the Software without restriction, including without limitation the
 *  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 *  sell copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 *  IN THE SOFTWARE.
 */

// Freemarker: 29 Oct 2021, 17:45:58 ch/raffael/meldioc/library/http/server/undertow/routing/RoutingBuilder.java.ftl
package ch.raffael.meldioc.library.http.server.undertow.routing;

import ch.raffael.meldioc.library.codec.ContentType;
import ch.raffael.meldioc.library.http.server.undertow.codec.EmptyBody;
import ch.raffael.meldioc.library.http.server.undertow.util.HttpMethod;
import io.undertow.server.HttpHandler;
import io.vavr.collection.HashSet;
import io.vavr.control.Either;

import java.util.function.Function;

// vagen Thu Nov 17 08:23:55 CET 2022
/**
 * Builder for child DSL frames (path segments and captures).
 */
class RoutingBuilder {

  private static final String STRING_NAME = "string";
  private static final String INT_NAME = "int";

  private final Frame initiatingFrame;

  private RoutingBuilder(Frame initiatingFrame) {
    this.initiatingFrame = initiatingFrame;
  }

  static RoutingBuilder.InitialFragment begin(Frame frame) {
    return new RoutingBuilder(frame).new InitialFragment();
  }

  private static String captureName(Converter<?> converter) {
    return "capture";
  }

  static abstract class AbstractFragment {

    abstract Frame resolve();

    String name() {
      StringBuilder buf = new StringBuilder();
      name(buf);
      return buf.toString();
    }

    abstract void name(StringBuilder buf);

    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder("RoutingBuilder[");
      name(buf);
      return buf.append("]").toString();
    }
  }

  public final class InitialFragment extends AbstractFragment {
    private InitialFragment() {
    }

    public Fragment0 path(String path) {
      return new Fragment0(this, path);
    }

    public Fragment1<String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment1<String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment1<Integer> captureInt() {
      return capture(INT_NAME, Converter.asInt());
    }

    public Fragment1<Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <T> Fragment1<T> capture(Converter<? extends T> converter) {
      return capture(captureName(converter), converter);
    }

    public <T> Fragment1<T> capture(String name, Converter<? extends T> converter) {
      return new Fragment1<>(this, new Capture.Attachment<>(name, converter));
    }

    @Override
    Frame resolve() {
      return initiatingFrame;
    }

    @Override
    void name(StringBuilder buf) {
      buf.append("/");
    }
  }

  static abstract class FollowupFragment extends AbstractFragment {
    private final AbstractFragment parent;
    private final Either<String, Capture.Attachment<?>> segment;

    private FollowupFragment(AbstractFragment parent, String path) {
      this.parent = parent;
      this.segment = Either.left(path);
    }

    private FollowupFragment(AbstractFragment parent, Capture.Attachment<?> capture) {
      this.parent = parent;
      this.segment = Either.right(capture);
    }

    @Override
    Frame resolve() {
      var frame = parent.resolve();
      return segment.fold(frame::pathChild, frame::captureChild);
    }

    @Override
    void name(StringBuilder buf) {
      buf.append("/");
      segment.fold(
          buf::append,
          c -> buf.append("{").append(c.name()).append("}"));
    }

    public EndpointBuilder.Method get() {
      return resolve().endpoint(HashSet.of(HttpMethod.GET));
    }

    public EndpointBuilder.Method head() {
      return resolve().endpoint(HashSet.of(HttpMethod.HEAD));
    }

    public EndpointBuilder.Method post() {
      return resolve().endpoint(HashSet.of(HttpMethod.POST));
    }

    public EndpointBuilder.Method put() {
      return resolve().endpoint(HashSet.of(HttpMethod.PUT));
    }

    public EndpointBuilder.Method patch() {
      return resolve().endpoint(HashSet.of(HttpMethod.PATCH));
    }

    public EndpointBuilder.Method delete() {
      return resolve().endpoint(HashSet.of(HttpMethod.DELETE));
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(ContentType contentType, String resource) {
      return ResourceLoader.apply(resolve(), contentType, StackWalker.getInstance().getCallerClass(), resource);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(ContentType contentType, Class<?> resourceClass, String resource) {
      return ResourceLoader.apply(resolve(), contentType, resourceClass, resource);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(String contentType, String resource) {
      return ResourceLoader.apply(resolve(), contentType, StackWalker.getInstance().getCallerClass(), resource);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(String contentType, Class<?> resourceClass, String resource) {
      return ResourceLoader.apply(resolve(), contentType, resourceClass, resource);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(ContentType contentType, String resource,
        Actions.Action1<? super byte[], ? extends byte[]> processor) {
      return ResourceLoader.apply(resolve(), contentType, StackWalker.getInstance().getCallerClass(), resource, processor);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(ContentType contentType, Class<?> resourceClass, String resource,
        Actions.Action1<? super byte[], ? extends byte[]> processor) {
      return ResourceLoader.apply(resolve(), contentType, resourceClass, resource, processor);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(String contentType, String resource,
        Actions.Action1<? super byte[], ? extends byte[]> processor) {
      return ResourceLoader.apply(resolve(), contentType, StackWalker.getInstance().getCallerClass(), resource, processor);
    }

    public EndpointBuilder<EmptyBody, byte[]> resource(String contentType, Class<?> resourceClass, String resource,
        Actions.Action1<? super byte[], ? extends byte[]> processor) {
      return ResourceLoader.apply(resolve(), contentType, resourceClass, resource, processor);
    }

    public void handler(Function<? super HttpHandler, ? extends HttpHandler> handler) {
      resolve().handler(handler);
    }

    public void merge(RoutingDefinition that) {
      resolve().merge(that.rootFrame);
    }
  }

  public final class Fragment0 extends FollowupFragment {

    private Fragment0(AbstractFragment parent, String path) {
      super(parent, path);
    }

    private Fragment0(AbstractFragment parent, Capture.Attachment<?> capture) {
      super(parent, capture);
    }

    public Fragment0 path(String path) {
      return Paths.empty(path) ? this : new Fragment0(this, path);
    }

    public Fragment1<String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment1<String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment1<Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment1<Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment1<TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment1<TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment1<>(this, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block0 block) {
      resolve().run(block);
    }
  }

  public final class Fragment1<T1> extends FollowupFragment {
    private final Blocks.Curry1<? extends T1> curry;

    private Fragment1(AbstractFragment parent, Blocks.Curry1<? extends T1> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment1(AbstractFragment parent, Capture.Attachment<? extends T1> capture) {
      super(parent, capture);
      this.curry = Blocks.curry(capture);
    }

    public Fragment1<T1> path(String path) {
      return Paths.empty(path) ? this : new Fragment1<>(this, curry, path);
    }

    public Fragment2<T1, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment2<T1, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment2<T1, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment2<T1, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment2<T1, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment2<T1, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment2<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block1<? super T1> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment2<T1, T2> extends FollowupFragment {
    private final Blocks.Curry2<? extends T1, ? extends T2> curry;

    private Fragment2(AbstractFragment parent, Blocks.Curry2<? extends T1, ? extends T2> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment2(AbstractFragment parent, Blocks.Curry1<? extends T1> curry, Capture.Attachment<? extends T2> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment2<T1, T2> path(String path) {
      return Paths.empty(path) ? this : new Fragment2<>(this, curry, path);
    }

    public Fragment3<T1, T2, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment3<T1, T2, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment3<T1, T2, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment3<T1, T2, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment3<T1, T2, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment3<T1, T2, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment3<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block2<? super T1, ? super T2> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment3<T1, T2, T3> extends FollowupFragment {
    private final Blocks.Curry3<? extends T1, ? extends T2, ? extends T3> curry;

    private Fragment3(AbstractFragment parent, Blocks.Curry3<? extends T1, ? extends T2, ? extends T3> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment3(AbstractFragment parent, Blocks.Curry2<? extends T1, ? extends T2> curry, Capture.Attachment<? extends T3> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment3<T1, T2, T3> path(String path) {
      return Paths.empty(path) ? this : new Fragment3<>(this, curry, path);
    }

    public Fragment4<T1, T2, T3, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment4<T1, T2, T3, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment4<T1, T2, T3, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment4<T1, T2, T3, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment4<T1, T2, T3, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment4<T1, T2, T3, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment4<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block3<? super T1, ? super T2, ? super T3> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment4<T1, T2, T3, T4> extends FollowupFragment {
    private final Blocks.Curry4<? extends T1, ? extends T2, ? extends T3, ? extends T4> curry;

    private Fragment4(AbstractFragment parent, Blocks.Curry4<? extends T1, ? extends T2, ? extends T3, ? extends T4> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment4(AbstractFragment parent, Blocks.Curry3<? extends T1, ? extends T2, ? extends T3> curry, Capture.Attachment<? extends T4> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment4<T1, T2, T3, T4> path(String path) {
      return Paths.empty(path) ? this : new Fragment4<>(this, curry, path);
    }

    public Fragment5<T1, T2, T3, T4, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment5<T1, T2, T3, T4, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment5<T1, T2, T3, T4, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment5<T1, T2, T3, T4, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment5<T1, T2, T3, T4, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment5<T1, T2, T3, T4, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment5<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block4<? super T1, ? super T2, ? super T3, ? super T4> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment5<T1, T2, T3, T4, T5> extends FollowupFragment {
    private final Blocks.Curry5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> curry;

    private Fragment5(AbstractFragment parent, Blocks.Curry5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment5(AbstractFragment parent, Blocks.Curry4<? extends T1, ? extends T2, ? extends T3, ? extends T4> curry, Capture.Attachment<? extends T5> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment5<T1, T2, T3, T4, T5> path(String path) {
      return Paths.empty(path) ? this : new Fragment5<>(this, curry, path);
    }

    public Fragment6<T1, T2, T3, T4, T5, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment6<T1, T2, T3, T4, T5, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment6<T1, T2, T3, T4, T5, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment6<T1, T2, T3, T4, T5, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment6<T1, T2, T3, T4, T5, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment6<T1, T2, T3, T4, T5, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment6<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block5<? super T1, ? super T2, ? super T3, ? super T4, ? super T5> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment6<T1, T2, T3, T4, T5, T6> extends FollowupFragment {
    private final Blocks.Curry6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> curry;

    private Fragment6(AbstractFragment parent, Blocks.Curry6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment6(AbstractFragment parent, Blocks.Curry5<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5> curry, Capture.Attachment<? extends T6> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment6<T1, T2, T3, T4, T5, T6> path(String path) {
      return Paths.empty(path) ? this : new Fragment6<>(this, curry, path);
    }

    public Fragment7<T1, T2, T3, T4, T5, T6, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment7<T1, T2, T3, T4, T5, T6, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment7<T1, T2, T3, T4, T5, T6, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment7<T1, T2, T3, T4, T5, T6, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment7<T1, T2, T3, T4, T5, T6, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment7<T1, T2, T3, T4, T5, T6, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment7<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block6<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment7<T1, T2, T3, T4, T5, T6, T7> extends FollowupFragment {
    private final Blocks.Curry7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> curry;

    private Fragment7(AbstractFragment parent, Blocks.Curry7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment7(AbstractFragment parent, Blocks.Curry6<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6> curry, Capture.Attachment<? extends T7> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment7<T1, T2, T3, T4, T5, T6, T7> path(String path) {
      return Paths.empty(path) ? this : new Fragment7<>(this, curry, path);
    }

    public Fragment8<T1, T2, T3, T4, T5, T6, T7, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment8<T1, T2, T3, T4, T5, T6, T7, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment8<T1, T2, T3, T4, T5, T6, T7, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment8<T1, T2, T3, T4, T5, T6, T7, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment8<T1, T2, T3, T4, T5, T6, T7, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment8<T1, T2, T3, T4, T5, T6, T7, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment8<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block7<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment8<T1, T2, T3, T4, T5, T6, T7, T8> extends FollowupFragment {
    private final Blocks.Curry8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> curry;

    private Fragment8(AbstractFragment parent, Blocks.Curry8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment8(AbstractFragment parent, Blocks.Curry7<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7> curry, Capture.Attachment<? extends T8> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment8<T1, T2, T3, T4, T5, T6, T7, T8> path(String path) {
      return Paths.empty(path) ? this : new Fragment8<>(this, curry, path);
    }

    public Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, String> captureString() {
      return capture(STRING_NAME, Converter.asString());
    }

    public Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, String> captureString(String name) {
      return capture(name, Converter.asString());
    }

    public Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, Integer> captureInt() {
      return capture(STRING_NAME, Converter.asInt());
    }

    public Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, Integer> captureInt(String name) {
      return capture(name, Converter.asInt());
    }

    public <TC> Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, TC> capture(Converter<? extends TC> converter) {
      return capture(captureName(converter), converter);
    }

    public <TC> Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, TC> capture(String name, Converter<? extends TC> converter) {
      return new Fragment9<>(this, curry, new Capture.Attachment<>(name, converter));
    }

    public void route(Blocks.Block8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> block) {
      resolve().run(curry.runnable(block));
    }
  }

  public final class Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, T9> extends FollowupFragment {
    private final Blocks.Curry9<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8, ? extends T9> curry;

    private Fragment9(AbstractFragment parent, Blocks.Curry9<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8, ? extends T9> curry, String path) {
      super(parent, path);
      this.curry = curry;
    }

    private Fragment9(AbstractFragment parent, Blocks.Curry8<? extends T1, ? extends T2, ? extends T3, ? extends T4, ? extends T5, ? extends T6, ? extends T7, ? extends T8> curry, Capture.Attachment<? extends T9> capture) {
      super(parent, capture);
      this.curry = curry.append(capture);
    }

    public Fragment9<T1, T2, T3, T4, T5, T6, T7, T8, T9> path(String path) {
      return Paths.empty(path) ? this : new Fragment9<>(this, curry, path);
    }


    public void route(Blocks.Block9<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8, ? super T9> block) {
      resolve().run(curry.runnable(block));
    }
  }
}
