/*
 * @(#)Resources.java
 * Copyright © 2023 The authors and contributors of JHotDraw. MIT License.
 */
package org.jhotdraw8.application.resources;

import javafx.scene.Node;
import javafx.scene.control.ButtonBase;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuItem;
import javafx.scene.control.Tooltip;
import javafx.scene.input.KeyCombination;
import org.jhotdraw8.application.action.Action;
import org.jspecify.annotations.Nullable;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Formatter;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.NoSuchElementException;
import java.util.ResourceBundle;

/**
 * This is a convenience wrapper for accessing resources stored in a
 * ResourceBundle.
 * <p>
 * A resources object may reference a parent resources object using the resource
 * key "$parent".
 * <p>
 * <b>Placeholders</b><br>
 * On top of the functionality provided by ResourceBundle, a property value can
 * include text from another property, by specifying the desired property name
 * and format type between <code>"${"</code> and <code>"}"</code>.
 * <p>
 * For example, if there is a {@code "imagedir"} property with the value
 * {@code "/org/jhotdraw8/undo/images"}, then this could be used in an attribute
 * like this: <code>${imagedir}/editUndo.png</code>. This is resolved at
 * run-time as {@code /org/jhotdraw8/undo/images/editUndo.png}.
 * <p>
 * Property names in placeholders can contain modifiers. Modifiers are written
 * between @code "[$"} and {@code "]"}. Each modifier has a fallback chain.
 * <p>
 * For example, if the property name modifier {@code "os"} has the value "win",
 * and its fallback chain is {@code "mac","default"}, then the property name
 * <code>${preferences.text.[$os]}</code> is first evaluated to {@code
 * preferences.text.win}, and - if no property with this name exists - it is
 * evaluated to {@code preferences.text.mac}, and then to
 * {@code preferences.text.default}.
 * <p>
 * The property name modifier "os" is defined by default. It can assume the
 * values "win", "mac" and "other". Its fallback chain is "default".
 * <p>
 * The format type can be optionally specified after a comma. The following
 * format types are supported:
 * <ul>
 * <li>{@code string} This is the default format.</li>
 * <li>{@code accelerator} This format replaces all occurrences of the keywords
 * shift, control, ctrl, meta, alt, altGraph by properties which start with
 * {@code accelerator.}. For example, shift is replaced by
 * {@code accelerator.shift}.
 * </li>
 * </ul>
 *
 * @author Werner Randelshofer
 */
public interface Resources {
    String PARENT_RESOURCE_KEY = "$parent";

    /**
     * Adds a decoder.
     *
     * @param decoder the resource decoder
     */
    static void addDecoder(ResourceDecoder decoder) {
        ResourcesHelper.decoders.add(decoder);
    }

    /**
     * Puts a property name modifier along with a fallback chain.
     *
     * @param name          The name of the modifier.
     * @param fallbackChain The fallback chain of the modifier.
     */
    static void putPropertyNameModifier(String name, String... fallbackChain) {
        ResourcesHelper.propertyNameModifiers.put(name, fallbackChain);
    }

    /**
     * Removes a property name modifier.
     *
     * @param name The name of the modifier.
     */
    static void removePropertyNameModifier(String name) {
        ResourcesHelper.propertyNameModifiers.remove(name);
    }

    static Resources getResources(String moduleName, String resourceBundle) {
        try {
            Class<?> clazz = Class.forName("org.jhotdraw8.application.resources.ModulepathResources");
            Method method = clazz.getMethod("getResources", String.class, String.class);
            return (Resources) method.invoke(null, moduleName, resourceBundle);
        } catch (IllegalAccessException | NoSuchMethodException | ClassNotFoundException e) {
            return ClasspathResources.getResources(resourceBundle);
        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            if (cause instanceof MissingResourceException) {
                throw (MissingResourceException) cause;
            }
            return ClasspathResources.getResources(resourceBundle);
        }
    }

    ResourceBundle asResourceBundle();

    default void configureAction(Action action, String argument) {
        configureAction(action, argument, getBaseClass());
    }

    default void configureAction(Action action, String argument, Class<?> baseClass) {
        action.set(Action.LABEL, getTextProperty(argument));
        String shortDescription = getToolTipTextProperty(argument);
        if (shortDescription != null && !shortDescription.isEmpty()) {
            action.set(Action.SHORT_DESCRIPTION, shortDescription);
        }
        action.set(Action.ACCELERATOR_KEY, getAcceleratorProperty(argument));
        action.set(Action.MNEMONIC_KEY, getMnemonicProperty(argument));
        action.set(Action.SMALL_ICON, getSmallIconProperty(argument, baseClass));
        action.set(Action.LARGE_ICON_KEY, getLargeIconProperty(argument, baseClass));
    }

    default void configureButton(ButtonBase button, String argument) {
        configureButton(button, argument, getBaseClass());
    }

    default void configureButton(ButtonBase button, String argument, Class<?> baseClass) {
        button.setText(getTextProperty(argument));
        //button.setACCELERATOR_KEY, getAcceleratorProperty(argument));
        //action.putValue(Action.MNEMONIC_KEY, new Integer(getMnemonicProperty(argument)));
        button.setGraphic(getLargeIconProperty(argument, baseClass));
        button.setTooltip(new Tooltip(getToolTipTextProperty(argument)));
    }

    /**
     * Configures a menu item with a text, an accelerator, a mnemonic and a menu
     * icon.
     *
     * @param menu     the menu
     * @param argument the argument
     */
    default void configureMenu(Menu menu, String argument) {
        menu.setText(getTextProperty(argument));
        menu.setAccelerator(getAcceleratorProperty(argument));
        menu.setGraphic(getSmallIconProperty(argument, getBaseClass()));
    }

    /**
     * Configures a menu item with a text, an accelerator, a mnemonic and a menu
     * icon.
     *
     * @param menu     the menu item
     * @param argument the argument
     */
    default void configureMenuItem(MenuItem menu, String argument) {
        menu.setText(getTextProperty(argument));
        menu.setAccelerator(getAcceleratorProperty(argument));
        menu.setGraphic(getSmallIconProperty(argument, getBaseClass()));
    }

    default void configureToolBarButton(ButtonBase button, String argument) {
        configureToolBarButton(button, argument, getBaseClass());
    }

    default void configureToolBarButton(ButtonBase button, String argument, Class<?> baseClass) {
        Node icon = getLargeIconProperty(argument, baseClass);
        if (icon != null) {
            button.setGraphic(getLargeIconProperty(argument, baseClass));
            button.setText(null);
        } else {
            button.setGraphic(null);
            button.setText(getTextProperty(argument));
        }
        button.setTooltip(new Tooltip(getToolTipTextProperty(argument)));
    }

    boolean containsKey(String key);

    /**
     * Returns a formatted string using {@link Formatter}.
     *
     * @param key       the key
     * @param arguments the arguments
     * @return formatted String
     */
    default String format(String key, Object... arguments) {
        return new Formatter(getLocale()).format(getString(key), arguments).toString();
    }

    /**
     * Returns a formatted string using {@link MessageFormat}.
     *
     * @param key       the key
     * @param arguments the arguments
     * @return formatted String
     */
    default String messageFormat(String key, Object... arguments) {
        return new MessageFormat(getString(key), getLocale()).format(arguments);
    }

    /**
     * Gets a KeyStroke for a JavaBeans "accelerator" property from the
     * ResourceBundle.
     * <BR>Convenience method.
     *
     * @param key The key of the property. This method adds ".accelerator" to
     *            the key.
     * @return <code>javax.swing.KeyStroke.getKeyStroke(value)</code>. Returns
     * null if the property is missing.
     */
    default @Nullable KeyCombination getAcceleratorProperty(String key) {
        return getKeyCombination(key + ".accelerator");
    }

    Class<?> getBaseClass();


    /**
     * Returns a java.lang.Module if the resources have been created
     * with named Java module on a Java VM that supports modules.
     *
     * @return java.lang.Module
     */
    Object getModule();

    String getBaseName();

    /**
     * Returns a formatted string using javax.text.MessageFormat.
     *
     * @param key       the key
     * @param arguments the arguments
     * @return formatted String
     */
    default String getFormatted(String key, Object... arguments) {
        return MessageFormat.format(getString(key), arguments);
    }

    /**
     * Get an Integer from the ResourceBundle.
     * <br>Convenience method to save casting.
     *
     * @param key The key of the property.
     * @return The value of the property. Returns -1 if the property is missing.
     */
    default Integer getInteger(String key) {
        try {
            return Integer.valueOf(getString(key));
        } catch (MissingResourceException e) {
            ResourcesHelper.LOG.warning("ClasspathResources[" + getBaseName() + "] \"" + key + "\" not found.");
            return -1;
        }
    }

    /**
     * Get a KeyStroke from the ResourceBundle.
     * <BR>Convenience method.
     *
     * @param key The key of the property.
     * @return <code>javax.swing.KeyStroke.getKeyStroke(value)</code>. Returns
     * null if the property is missing.
     */
    default @Nullable KeyCombination getKeyCombination(String key) {
        KeyCombination ks = null;
        String s = getString(key);
        try {
            ks = (s == null || s.isEmpty()) ? null : KeyCombination.valueOf(translateKeyStrokeToKeyCombination(s));
        } catch (NoSuchElementException | StringIndexOutOfBoundsException e) {
            throw new InternalError(key + "=" + s, e);
        }
        return ks;
    }


    /**
     * Get a large image icon from the ResourceBundle for use on a
     * {@code JButton}.
     * <br>Convenience method .
     *
     * @param key       The key of the property. This method appends ".largeIcon" to
     *                  the key.
     * @param baseClass the base class used to retrieve the image resource
     * @return The value of the property. Returns null if the property is
     * missing.
     */
    default @Nullable Node getLargeIconProperty(String key, Class<?> baseClass) {
        return ResourcesHelper.getIconProperty(this, key, ".largeIcon", baseClass);
    }

    Locale getLocale();

    /**
     * Get a Mnemonic from the ResourceBundle.
     * <br>Convenience method.
     *
     * @param key The key of the property.
     * @return The first char of the value of the property. Returns '\0' if the
     * property is missing.
     */
    default char getMnemonic(String key) {
        String s = getString(key);
        return (s == null || s.isEmpty()) ? '\0' : s.charAt(0);
    }

    /**
     * Gets a char for a JavaBeans "mnemonic" property from the ResourceBundle.
     * <br>Convenience method.
     *
     * @param key The key of the property. This method appends ".mnemonic" to
     *            the key.
     * @return The first char of the value of the property. Returns '\0' if the
     * property is missing.
     */
    default @Nullable KeyCombination getMnemonicProperty(String key) {
        String s;
        try {
            s = getString(key + ".mnemonic");
        } catch (MissingResourceException e) {
            ResourcesHelper.LOG.warning("Warning ClasspathResources[" + getBaseName() + "] \"" + key + ".mnemonic\" not found.");
            s = null;
        }
        return (s == null || s.isEmpty()) ? null : KeyCombination.valueOf(s);
    }

    /**
     * Get a small image icon from the ResourceBundle for use on a
     * {@code JMenuItem}.
     * <br>Convenience method .
     *
     * @param key       The key of the property. This method appends ".smallIcon" to
     *                  the key.
     * @param baseClass the base class used to retrieve the image resource
     * @return The value of the property. Returns null if the property is
     * missing.
     */
    default @Nullable Node getSmallIconProperty(String key, Class<?> baseClass) {
        return ResourcesHelper.getIconProperty(this, key, ".smallIcon", baseClass);
    }

    String getString(String s);

    /**
     * Get a String for a JavaBeans "text" property from the ResourceBundle.
     * <br>Convenience method.
     *
     * @param key The key of the property. This method appends ".text" to the
     *            key.
     * @return The ToolTip. Returns null if no tooltip is defined.
     */
    default @Nullable String getTextProperty(String key) {
        try {
            String value = getString(key + ".text");
            return value;
        } catch (MissingResourceException e) {
            ResourcesHelper.LOG.warning("Warning ClasspathResources[" + getBaseName() + "] \"" + key + ".text\" not found.");
            return null;
        }
    }

    /**
     * Get a String for a JavaBeans "toolTipText" property from the
     * ResourceBundle.
     * <br>Convenience method.
     *
     * @param key The key of the property. This method appends ".toolTipText" to
     *            the key.
     * @return The ToolTip. Returns null if no tooltip is defined.
     */
    default @Nullable String getToolTipTextProperty(String key) {
        try {
            String value = getString(key + ".toolTipText");
            return value;
        } catch (MissingResourceException e) {
            ResourcesHelper.LOG.warning("Resources[" + getBaseName() + "] \"" + key + ".toolTipText\" not found.");
            return null;
        }
    }

    default String substitutePlaceholders(String key, String value) throws MissingResourceException {

        // Substitute placeholders in the value
        for (int p1 = value.indexOf("${"); p1 != -1; p1 = value.indexOf("${")) {
            int p2 = value.indexOf('}', p1 + 2);
            if (p2 < 0) {
                break;
            }

            String placeholderKey = value.substring(p1 + 2, p2);
            String placeholderFormat;
            int p3 = placeholderKey.indexOf(',');
            if (p3 != -1) {
                placeholderFormat = placeholderKey.substring(p3 + 1);
                placeholderKey = placeholderKey.substring(0, p3);
            } else {
                placeholderFormat = "string";
            }
            ArrayList<String> fallbackKeys = new ArrayList<>();
            ResourcesHelper.generateFallbackKeys(placeholderKey, fallbackKeys);

            String placeholderValue = null;
            for (String fk : fallbackKeys) {
                try {
                    placeholderValue = getString(fk);
                    break;
                } catch (MissingResourceException e) {
                }
            }
            if (placeholderValue == null) {
                throw new MissingResourceException("Could not find the placeholder value for key=\"" + key + "\", value=\"" + value + "\", placeholderKey=\"" + placeholderKey + "\".", getBaseName(), key);
            }

            // Do post-processing depending on placeholder format
            if ("accelerator".equals(placeholderFormat)) {
                // Localize the keywords shift, control, ctrl, meta, alt, altGraph
                StringBuilder b = new StringBuilder();
                for (String s : placeholderValue.split(" ")) {
                    if (ResourcesHelper.acceleratorKeys.contains(s)) {
                        b.append(getString("accelerator." + s));
                    } else {
                        b.append(s);
                    }
                }
                placeholderValue = b.toString();
            }

            // Insert placeholder value into value
            value = value.substring(0, p1) + placeholderValue + value.substring(p2 + 1);
        }

        return value;

    }

    /**
     * Translate a String defining a {@code javax.swing.KeyStroke} into a String
     * for {@code javafx.input.KeyCombination}.
     *
     * @param s The KeyStroke String
     * @return The KeyCombination String
     */
    default @Nullable String translateKeyStrokeToKeyCombination(@Nullable String s) {
        if (s != null) {
            s = s.replace("ctrl ", "Ctrl+");
            s = s.replace("meta ", "Meta+");
            s = s.replace("alt ", "Alt+");
            s = s.replace("shift ", "Shift+");
        }
        return s;
    }

    @Nullable
    Object handleGetObjectRecursively(String key);

    Enumeration<String> getKeys();

    @Nullable
    Resources getParent();

    void setParent(@Nullable Resources newParent);

    default Resources getRoot() {
        Resources root = this;
        while (root.getParent() != null) {
            root = root.getParent();
        }
        return root;
    }
}
