/*
 * @(#)DefaultableStyleableKey.java
 * Copyright © 2023 The authors and contributors of JHotDraw. MIT License.
 */
package org.jhotdraw8.draw.key;

import org.jhotdraw8.base.converter.Converter;
import org.jhotdraw8.css.converter.CssConverter;
import org.jhotdraw8.css.value.CssDefaultableValue;
import org.jhotdraw8.draw.css.converter.DefaultableValueCssConverter;
import org.jhotdraw8.fxbase.styleable.WritableStyleableMapAccessor;
import org.jhotdraw8.fxcollection.typesafekey.NonNullKey;
import org.jspecify.annotations.Nullable;

import java.lang.reflect.Type;

/**
 * DefaultableStyleableKey.
 *
 * @param <T> the value type
 * @author Werner Randelshofer
 */
public class DefaultableStyleableKey<T> extends AbstractStyleableKey<CssDefaultableValue<T>>
        implements WritableStyleableMapAccessor<CssDefaultableValue<T>>,
        DefaultableStyleableMapAccessor<T>, NonNullKey<CssDefaultableValue<T>> {


    private final Converter<CssDefaultableValue<T>> converter;
    private final T initialValue;


    /**
     * Creates a new instance with the specified name, mask and default value.
     *
     * @param name                   The name of the key.
     * @param type                   The full type
     * @param converter              String converter for a list element
     * @param initialDefaultingValue The default value.
     */
    public DefaultableStyleableKey(String name, Type type, CssConverter<T> converter,
                                   CssDefaultableValue<T> initialDefaultingValue,
                                   @Nullable T initialValue) {
        super(name, type, initialDefaultingValue);
        this.initialValue = initialValue;

        this.converter = new DefaultableValueCssConverter<>(converter);
    }

    @Override
    public Converter<CssDefaultableValue<T>> getCssConverter() {
        return converter;
    }

    /**
     * Returns the initial value of the attribute.
     * <p>
     * We use the definition from CSS initial value:
     * <p>
     * "Each property has an initial value, defined in
     * the property's definition table. If the property
     * is not an inherited property, and the cascade does not
     * result in a value, then the specified value of the
     * property is its initial value."
     * <p>
     * We intentionally do <b>not</b> use the definition from SVG
     * initial value:
     * <p>
     * <s>"The initial value of an attribute or property is
     * the value used when that attribute or property is not
     * specified, or when it has an invalid value."</s>
     * <p>
     * References:
     * <dl>
     *     <dt>CSS Cascading and Inheritance Level 4, Chapter 7.1 Initial Values></dt>
     *     <dd><a href="https://www.w3.org/TR/css-cascade-4/#initial-values">w3.org</a></dd>
     *
     *     <dt>SVG, Chapter 4: Basic Data Types and Interfaces, 4.1 Definitions, Initial Value</dt>
     *     <dd><a href="https://www.w3.org/TR/SVG/types.html#definitions">w3.org</a></dd>
     * </dl>
     *
     * @return the initial value.
     */
    @Override
    public T getInitialValue() {
        return initialValue;
    }
}
