/*
 * @(#)AbstractDockParent.java
 * Copyright © 2023 The authors and contributors of JHotDraw. MIT License.
 */
package org.jhotdraw8.fxcontrols.dock;

import javafx.beans.property.BooleanProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.collections.FXCollections;
import javafx.collections.ListChangeListener;
import javafx.collections.ObservableList;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.VPos;
import javafx.scene.Node;
import javafx.scene.Parent;
import javafx.scene.layout.Region;

import java.util.List;


/**
 * Abstract base class for implementations of {@link DockParent}.
 */
public abstract class AbstractDockParent
        extends Region
        implements DockParent {
    @SuppressWarnings("this-escape")
    protected final BooleanProperty showing = new SimpleBooleanProperty(this, SHOWING_PROPERTY);
    @SuppressWarnings("this-escape")
    protected final BooleanProperty editable = new SimpleBooleanProperty(this, EDITABLE_PROPERTY, true);
    @SuppressWarnings("this-escape")
    protected final ObjectProperty<DockParent> dockParent = new SimpleObjectProperty<>(this, DOCK_PARENT_PROPERTY);
    protected final ObservableList<DockChild> dockChildren = FXCollections.observableArrayList();

    public AbstractDockParent() {
        dockChildren.addListener((ListChangeListener.Change<? extends DockChild> change) -> {
            while (change.next()) {
                for (DockChild removed : change.getRemoved()) {
                    removed.setDockParent(null);
                }
                for (DockChild added : change.getAddedSubList()) {
                    if (added.getDockParent() != null) {
                        added.getDockParent().getDockChildren().remove(added);
                    }
                    added.setDockParent(this);
                }
            }
        });
    }

    @Override
    public ObjectProperty<DockParent> dockParentProperty() {
        return dockParent;
    }


    @Override
    public ObservableList<DockChild> getDockChildren() {
        return dockChildren;
    }

    @Override
    public Parent getNode() {
        return this;
    }

    @Override
    public BooleanProperty showingProperty() {
        return showing;
    }

    @Override
    public BooleanProperty editableProperty() {
        return editable;
    }

    @Override
    protected void layoutChildren() {
        List<Node> managed = getManagedChildren();
        final double width = getWidth();
        double height = getHeight();
        Insets insets = getInsets();
        double top = insets.getTop();
        double right = insets.getRight();
        double left = insets.getLeft();
        double bottom = insets.getBottom();
        double contentWidth = width - left - right;
        double contentHeight = height - top - bottom;
        double baselineOffset = 0;
        HPos alignHpos = HPos.LEFT;
        VPos alignVpos = VPos.TOP;

        for (Node child : managed) {
            layoutInArea(child, left, top,
                    contentWidth, contentHeight,
                    baselineOffset, null,
                    alignHpos,
                    alignVpos);
        }
    }
}
