/*
 * Decompiled with CFR 0.152.
 */
package org.monte.media.screenrecorder;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicLong;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.BooleanControl;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.TargetDataLine;
import org.monte.media.av.Buffer;
import org.monte.media.av.Format;
import org.monte.media.av.codec.audio.AudioFormatKeys;
import org.monte.media.math.Rational;

class AudioGrabber
implements Runnable,
AutoCloseable {
    private final TargetDataLine line;
    private final BlockingQueue<Buffer> queue;
    private final Integer audioTrack;
    private final Long startTime;
    private Long totalSampleCount;
    private ScheduledFuture<?> future;
    private Long sequenceNumber;
    private float audioLevelLeft = -1.0f;
    private float audioLevelRight = -1.0f;
    private final AtomicLong stopTime = new AtomicLong(Long.MAX_VALUE);

    public AudioGrabber(Mixer mixer, Format audioFormat, int audioTrack, long startTime, BlockingQueue<Buffer> queue) throws LineUnavailableException {
        this.audioTrack = audioTrack;
        this.queue = queue;
        this.startTime = startTime;
        DataLine.Info lineInfo = new DataLine.Info(TargetDataLine.class, AudioFormatKeys.toAudioFormat((Format)audioFormat));
        this.line = this.initializeAudioLine(mixer, lineInfo);
        this.configureAudioLine();
    }

    private TargetDataLine initializeAudioLine(Mixer mixer, DataLine.Info lineInfo) throws LineUnavailableException {
        if (mixer != null) {
            return (TargetDataLine)mixer.getLine(lineInfo);
        }
        return (TargetDataLine)AudioSystem.getLine(lineInfo);
    }

    private void configureAudioLine() throws LineUnavailableException {
        this.unmuteLine();
        this.setMinimumVolume();
        this.line.open();
        this.line.start();
    }

    private void unmuteLine() {
        try {
            BooleanControl muteControl = (BooleanControl)this.line.getControl(BooleanControl.Type.MUTE);
            muteControl.setValue(false);
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    private void setMinimumVolume() {
        try {
            FloatControl volumeControl = (FloatControl)this.line.getControl(FloatControl.Type.VOLUME);
            volumeControl.setValue(Math.max(volumeControl.getValue(), 0.2f));
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void setFuture(ScheduledFuture<?> future) {
        this.future = future;
    }

    @Override
    public void close() {
        this.line.close();
    }

    public void setStopTime(long newValue) {
        this.stopTime.set(newValue);
    }

    public long getStopTime() {
        return this.stopTime.get();
    }

    @Override
    public void run() {
        Buffer buf = new Buffer();
        AudioFormat lineFormat = this.line.getFormat();
        buf.format = AudioFormatKeys.fromAudioFormat((AudioFormat)lineFormat).append(new Object[]{AudioFormatKeys.SilenceBugKey, true});
        int bufferSize = this.calculateBufferSize(lineFormat);
        byte[] bdat = new byte[bufferSize];
        buf.data = bdat;
        int count = this.line.read(bdat, 0, bdat.length);
        if (count > 0) {
            this.processAudioData(buf, lineFormat, bdat, count);
        }
    }

    private int calculateBufferSize(AudioFormat lineFormat) {
        int bufferSize = lineFormat.getFrameSize() * (int)lineFormat.getSampleRate();
        if (((int)lineFormat.getSampleRate() & 1) == 0) {
            bufferSize /= 2;
        }
        return bufferSize;
    }

    private void processAudioData(Buffer buf, AudioFormat lineFormat, byte[] bdat, Integer count) {
        this.computeAudioLevel(bdat, count, lineFormat);
        this.setBufferProperties(buf, lineFormat, count);
        if (this.isRecordingComplete(buf)) {
            this.truncateBuffer(buf, lineFormat);
            this.future.cancel(false);
        }
        if (buf.sampleCount > 0) {
            this.enqueueBuffer(buf);
        }
        this.totalSampleCount = this.totalSampleCount + (long)buf.sampleCount;
    }

    private void setBufferProperties(Buffer buf, AudioFormat lineFormat, int count) {
        Rational sampleRate = Rational.valueOf((double)lineFormat.getSampleRate());
        Rational frameRate = Rational.valueOf((double)lineFormat.getFrameRate());
        buf.sampleCount = count / (lineFormat.getSampleSizeInBits() / 8 * lineFormat.getChannels());
        buf.sampleDuration = sampleRate.inverse();
        buf.offset = 0;
        Long l = this.sequenceNumber;
        this.sequenceNumber = this.sequenceNumber + 1L;
        buf.sequenceNumber = l;
        buf.length = count;
        buf.track = this.audioTrack;
        buf.timeStamp = new Rational(this.totalSampleCount.longValue(), 1L).divide(frameRate);
    }

    private boolean isRecordingComplete(Buffer buf) {
        Rational stopTS = new Rational(this.getStopTime() - this.startTime, 1000L);
        return buf.timeStamp.add(buf.sampleDuration.multiply((long)buf.sampleCount)).compareTo(stopTS) > 0;
    }

    private void truncateBuffer(Buffer buf, AudioFormat lineFormat) {
        Rational stopTS = new Rational(this.getStopTime() - this.startTime, 1000L);
        buf.sampleCount = Math.max(0, (int)Math.ceil(stopTS.subtract(buf.timeStamp).divide(buf.sampleDuration).floatValue()));
        buf.length = buf.sampleCount * (lineFormat.getSampleSizeInBits() / 8 * lineFormat.getChannels());
    }

    private void enqueueBuffer(Buffer buf) {
        try {
            this.queue.put(buf);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    private void computeAudioLevel(byte[] data, Integer length, AudioFormat format) {
        this.audioLevelRight = -1.0f;
        this.audioLevelLeft = -1.0f;
        if (format.getEncoding().equals(AudioFormat.Encoding.PCM_SIGNED)) {
            switch (format.getSampleSizeInBits()) {
                case 8: {
                    this.computeAudioLevel8Bit(data, length, format);
                    break;
                }
                case 16: {
                    this.computeAudioLevel16Bit(data, length, format);
                }
            }
        }
    }

    private void computeAudioLevel8Bit(byte[] data, Integer length, AudioFormat format) {
        switch (format.getChannels()) {
            case 1: {
                this.audioLevelLeft = this.computeAudioLevelSigned8(data, 0, length, format.getFrameSize());
                break;
            }
            case 2: {
                this.audioLevelLeft = this.computeAudioLevelSigned8(data, 0, length, format.getFrameSize());
                this.audioLevelRight = this.computeAudioLevelSigned8(data, 1, length, format.getFrameSize());
            }
        }
    }

    private void computeAudioLevel16Bit(byte[] data, Integer length, AudioFormat format) {
        if (format.isBigEndian()) {
            switch (format.getChannels()) {
                case 1: {
                    this.audioLevelLeft = this.computeAudioLevelSigned16BE(data, 0, length, format.getFrameSize());
                    break;
                }
                case 2: {
                    this.audioLevelLeft = this.computeAudioLevelSigned16BE(data, 0, length, format.getFrameSize());
                    this.audioLevelRight = this.computeAudioLevelSigned16BE(data, 2, length, format.getFrameSize());
                }
            }
        } else {
            switch (format.getChannels()) {
                case 1: {
                    this.audioLevelLeft = this.computeAudioLevelSigned16LE(data, 0, length, format.getFrameSize());
                    break;
                }
                case 2: {
                    this.audioLevelLeft = this.computeAudioLevelSigned16LE(data, 0, length, format.getFrameSize());
                    this.audioLevelRight = this.computeAudioLevelSigned16LE(data, 2, length, format.getFrameSize());
                }
            }
        }
    }

    private float computeAudioLevelSigned16BE(byte[] data, int offset, int length, int stride) {
        double sum = 0.0;
        for (int i = offset; i < length; i += stride) {
            int value = data[i] << 8 | data[i + 1] & 0xFF;
            sum += (double)(value * value);
        }
        double rms = Math.sqrt(sum / (double)((length - offset) / stride));
        return (float)(rms / 32768.0);
    }

    private float computeAudioLevelSigned16LE(byte[] data, int offset, int length, int stride) {
        double sum = 0.0;
        for (int i = offset; i < length; i += stride) {
            int value = data[i + 1] << 8 | data[i] & 0xFF;
            sum += (double)(value * value);
        }
        double rms = Math.sqrt(sum / (double)((length - offset) / stride));
        return (float)(rms / 32768.0);
    }

    private float computeAudioLevelSigned8(byte[] data, int offset, int length, int stride) {
        double sum = 0.0;
        for (int i = offset; i < length; i += stride) {
            byte value = data[i];
            if (value == -128) continue;
            sum += (double)(value * value);
        }
        double rms = Math.sqrt(sum / ((double)length / (double)stride));
        return (float)(rms / 128.0);
    }

    public float getAudioLevelLeft() {
        return this.audioLevelLeft;
    }

    public float getAudioLevelRight() {
        return this.audioLevelRight;
    }
}

