/* ____  ______________  ________________________  __________
 * \   \/   /      \   \/   /   __/   /      \   \/   /      \
 *  \______/___/\___\______/___/_____/___/\___\______/___/\___\
 *
 * The MIT License (MIT)
 *
 * Copyright 2024 Vavr, https://vavr.io
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package io.vavr.collection.euler;

import io.vavr.collection.Stream;
import org.junit.jupiter.api.Test;

import static io.vavr.collection.euler.Utils.factors;
import static org.assertj.core.api.Assertions.assertThat;

public class Euler12Test {

    /**
     * <strong>Problem 12: Highly divisible triangular number</strong>
     * <p>
     * The sequence of triangle numbers is generated by adding the natural
     * numbers. So the 7th triangle number would be 1 + 2 + 3 + 4 + 5 + 6 + 7 =
     * 28. The first ten terms would be:
     * <p>
     * 1, 3, 6, 10, 15, 21, 28, 36, 45, 55, ...
     * <p>
     * Let us list the factors of the first seven triangle numbers:
     * <pre>
     * 1: 1
     * 3: 1,3
     * 6: 1,2,3,6
     * 10: 1,2,5,10
     * 15: 1,3,5,15
     * 21: 1,3,7,21
     * 28: 1,2,4,7,14,28
     * </pre>
     * <p>
     * We can see that 28 is the first triangle number to have over five
     * divisors.
     * <p>
     * What is the value of the first triangle number to have over five hundred
     * divisors?
     */
    @Test
    public void shouldSolveProblem12() {
        assertThat(valueOfFirstTriangleNumberWithMoreDivisorsThan(5)).isEqualTo(28);
        assertThat(valueOfFirstTriangleNumberWithMoreDivisorsThan(500)).isEqualTo(76_576_500);
    }

    private static long valueOfFirstTriangleNumberWithMoreDivisorsThan(long divisorCount) {
        return triangleNumbers()
                .find(t -> divisorCount(t) > divisorCount)
                .get();
    }

    private static Stream<Long> triangleNumbers() {
        return Stream.from(1L).scanLeft(0L, (a, l) -> a + l);
    }

    private static long divisorCount(long number) {
        return factors(number).length();
    }
}
