/*
 * Copyright the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ch.rasc.jcentserverclient.models;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Request for batch operations.
 */
@JsonInclude(Include.NON_EMPTY)
@SuppressWarnings({ "hiding" })
public record BatchRequest(@JsonProperty("commands") List<Command> commands,
		@JsonProperty("parallel") Boolean parallel) {

	public static Builder builder() {
		return new Builder();
	}

	public static class Builder {

		private List<Command> commands;

		private Boolean parallel;

		public Builder commands(Command... commands) {
			if (commands == null || commands.length == 0) {
				throw new IllegalArgumentException("'commands' is required and cannot be null or empty");
			}
			this.commands = List.of(commands);
			return this;
		}

		public Builder commands(List<Command> commands) {
			if (commands == null || commands.isEmpty()) {
				throw new IllegalArgumentException("'commands' is required and cannot be null or empty");
			}
			this.commands = List.copyOf(commands);
			return this;
		}

		public Builder parallel(Boolean parallel) {
			this.parallel = parallel;
			return this;
		}

		public BatchRequest build() {
			return new BatchRequest(this.commands, this.parallel);
		}

	}
}
