package ch.rfin.util;

import java.util.*;
import java.util.stream.Collectors;
import java.util.function.*;

/**
 * Utilities for working with pairs.
 *
 * @author Christoffer Fink
 * @version 1.0.0
 */
public final class Pairs {

  /**
   * Converts a Map.Entry to a pair.
   * @return {@code (key, value)}.
   */
  public static <T,S> Pair<T,S> pairFrom(Map.Entry<T,S> entry) {
    return Pair.of(entry.getKey(), entry.getValue());
  }

  /**
   * Converts a list to a pair.
   * @return {@code (x0, x1)}.
   */
  public static <T> Pair<T,T> pairFrom(List<T> items) {
    return Pair.of(items.get(0), items.get(1));
  }

  // TODO: Decide whether an uneven number of items is really unacceptable.
  /**
   * Converts a list to a list of pairs.
   * @return {@code [(x0, x1), (x2, x3), ...]}.
   * @throws IllegalArgumentException if uneven number of items
   */
  public static <T> List<Pair<T,T>> pairs(List<T> items) {
    final int len = items.size();
    if (len == 0) {
      return Collections.emptyList();
    }
    if (len % 2 != 0) {
      throw new IllegalArgumentException(len + " is not an even number of items");
    }
    List<Pair<T,T>> result = new ArrayList<>(len/2);
    for (int i = 1; i < len; i+=2) {
      result.add(Pair.of(items.get(i-1), items.get(i)));
    }
    return result;
  }

  /**
   * Converts a Map to a list of pairs.
   * @return {@code [(k0, v0), (k1, v1), ...]}.
   */
  public static <K,V> List<Pair<K,V>> pairs(Map<K,V> items) {
    if (items.isEmpty()) {
      return Collections.emptyList();
    }
    return items.entrySet().stream().map(Pairs::pairFrom).collect(Collectors.toList());
  }

  /**
   * Converts a collection of pairs to a Map.
   * @return {@code [(k0, v0), (k1, v1), ...]}.
   */
  public static <K,V> Map<K,V> toMap(Collection<Pair<K,V>> items) {
    if (items.isEmpty()) {
      return Collections.emptyMap();
    }
    return items.stream().collect(Collectors.toMap(Pair::get_1, Pair::get_2));
  }

  /**
   * Takes a binary function and returns a function that is unary for pairs.
   * Allows {@code streamOfPairs.map(function(f))} instead of
   * {@code streamOfPairs.map(p -> f.apply(p._1, p._2))}.
   * @see ch.rfin.util.Pair#apply(BiFunction)
   */
  public static <T1,T2,R> Function<Pair<T1,T2>,R> function(final BiFunction<T1,T2,R> f) {
    return p -> f.apply(p._1, p._2);
  }

  /**
   * Takes a unary function from pairs and returns a binary function.
   */
  public static <T1,T2,R> BiFunction<T1,T2,R> biFunction(final Function<Pair<T1,T2>,R> f) {
    return (a,b) -> f.apply(Pair.of(a,b));
  }

  /**
   * Takes a binary predicate and returns a predicate that is unary for pairs.
   * Allows {@code streamOfPairs.filter(predicate(f))} instead of
   * {@code streamOfPairs.filter(p -> f.test(p._1, p._2))}.
   * @see ch.rfin.util.Pair#test(BiPredicate)
   */
  public static <T1,T2> Predicate<Pair<T1,T2>> predicate(final BiPredicate<T1,T2> f) {
    return p -> f.test(p._1, p._2);
  }

  /**
   * Takes a unary predicate from pairs and returns a binary predicate.
   */
  public static <T1,T2> BiPredicate<T1,T2> biPredicate(final Predicate<Pair<T1,T2>> f) {
    return (a,b) -> f.test(Pair.of(a,b));
  }

  /**
   * Takes a binary consumer and returns a consumer that is unary for pairs.
   * Allows {@code streamOfPairs.forEach(consumer(f))} instead of
   * {@code streamOfPairs.forEach(p -> f.accept(p._1, p._2))}.
   * @see ch.rfin.util.Pair#accept(BiConsumer)
   */
  public static <T1,T2> Consumer<Pair<T1,T2>> consumer(final BiConsumer<T1,T2> f) {
    return p -> f.accept(p._1, p._2);
  }

  /**
   * Takes a unary consumer of pairs and returns a binary consumer.
   */
  public static <T1,T2> BiConsumer<T1,T2> biConsumer(final Consumer<Pair<T1,T2>> f) {
    return (a,b) -> f.accept(Pair.of(a,b));
  }

  /**
   * Takes a function and returns a function to pairs, where the first element
   * is the original input and the second element is the mapping.
   * {@code (f: x -> y) -> (g: x -> (x,y))}
   */
  public static <T,R> Function<T,Pair<T,R>> withId(final Function<T,R> f) {
    return x -> Pair.of(x, f.apply(x));
  }

}
