package ch.sahits.game.graphic.display.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import ch.sahits.game.event.Event;
import ch.sahits.game.event.IEventListener;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;

/**
 * The CityView is a view model of a city. It provides access to the city
 * model as well as player specific aspects of the city, for example buildings,
 * ships and so forth.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 19, 2011
 *
 */
public class CityPlayerProxy implements IEventListener{
	private final ICity city;
	private final IPlayer player;
	/** List holding all ships currently present in the city */
	private List<IShip> shipList = new ArrayList<IShip>();
	/** List holding all ships owned by the player currently present in the city */
	private List<IShip> playerShips = new ArrayList<IShip>();
	/** Reference the active ship of the player */
	private IShip activeShip=null;
	/**
	 * Constructor initializing the city and the player
	 * @param city
	 * @param player
	 */
	public CityPlayerProxy(ICity city, IPlayer player) {
		super();
		this.city = city;
		this.player = player;
	}
	/**
	 * Constructor initializing the city and the player with one ship in port.
	 * @param city
	 * @param player
	 * @param activeShip
	 */
	public CityPlayerProxy(ICity city, IPlayer player, IShip activeShip) {
		this(city,player);
		this.activeShip = activeShip;
		shipList.add(activeShip);
		playerShips.add(activeShip);
	}
	/**
	 * Retrieve the players active ship
	 * @return
	 */
	public IShip getActiveShip() {
		return activeShip;
	}
	/**
	 * Activate a ship. If the ship is not one of the players or not in port
	 * an {@link IllegalArgumentException} will be thrown
	 * @param ship
	 * @throws IllegalArgumentException if the ship cannot be activated due to its non existence
	 */
	public void activateShip(IShip ship){
		for (IShip s : playerShips) {
			if (s.equals(ship)){
				activeShip=s;
				break; // there can only be one
			}
		}
		throw new IllegalArgumentException("The ship "+ship+" of the player is not in port");
	}
	/**
	 * Retrieve the list of all ships of the player currently in the port. The list is unmodifiable.
	 * @return unmodifiable list of the players ships
	 */
	public List<IShip> getPlayersShips(){
		return Collections.unmodifiableList(playerShips);
	}
	/**
	 * Handle all the events that are generated from outside (not by the player).
	 * Such events are arriving/leaving ships, price changes.
	 */
	@Override
	public void gameUpdate(Event e, Object eventNotice) {
		// TODO Auto-generated method stub
	}
	/**
	 * A ship arrives in the city
	 * @param ship
	 */
	public void arrive(IShip ship){
		shipList.add(ship);
		if (player.getFleet().contains(ship)){
			playerShips.add(ship);
			if (playerShips.size()==1){
				activeShip=ship;
			}
		}
	}
	/**
	 * Ship leave the city
	 * @param ship
	 */
	public void leave(IShip ship){
		if (activeShip==ship){
			activeShip=null;
		}
		playerShips.remove(ship);
		shipList.remove(ship);
	}
	public ICity getCity() {
		return city;
	}
	public IPlayer getPlayer() {
		return player;
	}
	
	
	
}
