package ch.sahits.game.graphic.display.util;

import java.awt.Point;
/**
 * This helper class lets you calculate the relative offset in an overlayed image.
 * This is needed if you place your image not at a fixed distance from the right border or the top.
 * As with different screen resolutions you would like it to have it placed 150px to the left of the
 * right border. What you actually want is not the offset of the image but a point within. Defining the
 * image you know the relative location within the image, but you need to know the absolute position
 * on the screen. This helper class lets you compute these coordinates.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Mar 4, 2011
 *
 */
public final class OffsetCalculator{
	// TODO needs test case
	private final int xOff;
	private final int yOff;
	/**
	 * Initialize the offset of the image. These are the absolute coordinates of the
	 * top left corner of the image on the screen.
	 * @param xOff Offset in direction X (to the right)
	 * @param yOff Offset in direction Y (down)
	 */
	public OffsetCalculator(int xOff, int yOff) {
		super();
		this.xOff = xOff;
		this.yOff = yOff;
	}
	/**
	 * Compute the absolute offset of the point X
	 * @param relativeXOffset relative offset (in relation to the image) of X
	 * @return absolute offset of X
	 */
	public int computeXOffset(int relativeXOffset){
		return relativeXOffset+xOff;
	}
	/**
	 * Compute the absolute offset of the point Y
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @return absolute offset of Y
	 */
	public int computeYOffset(int relativeYOffset){
		return relativeYOffset+yOff;
	}
	/**
	 * Compute the absolute offset of the point
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @return absolute offset
	 */
	public Point computeOffset(int relativeXOffset, int relativeYOffset){
		return new Point(computeXOffset(relativeXOffset),computeYOffset(relativeYOffset));
	}
	/**
	 * Compute the absolute offset of the point X in the case that the relative offset
	 * is scaled dynamically
	 * @param relativeXOffset relative offset (in relation to the image) of X
	 * @return absolute offset of X
	 */
	public int computeXOffset(int relativeXOffset, double xScale){
		return (int) Math.rint(relativeXOffset*xScale+xOff);
	}
	/**
	 * Compute the absolute offset of the point Y in the case that the relative offset
	 * is scaled dynamically
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @return absolute offset of Y
	 */
	public int computeYOffset(int relativeYOffset,double yScale){
		return (int) Math.rint(relativeYOffset*yScale+yOff);
	}
	/**
	 * Compute the absolute offset of the point in the case that the relative offsets are scaled
	 * dynamically.
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @param relativeYOffset relative offset (in relation to the image) of Y
	 * @return absolute offset
	 */
	public Point computeOffset(int relativeXOffset,double xScale, int relativeYOffset,double yScale){
		return new Point(computeXOffset(relativeXOffset,xScale),computeYOffset(relativeYOffset,yScale));
	}
}