package ch.sahits.game.graphic.display;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;

import ch.sahits.game.event.Event;
import ch.sahits.game.graphic.display.gameplay.GameStatus;
import ch.sahits.game.graphic.display.gameplay.MainMenu;
import ch.sahits.game.graphic.display.gameplay.MainView;
import ch.sahits.game.graphic.display.gameplay.MapView;
import ch.sahits.game.graphic.display.gameplay.SubMenu;
import ch.sahits.game.graphic.display.gameplay.ViewStatus;
import ch.sahits.game.graphic.display.notice.NoticeView;
import ch.sahits.game.graphic.image.DisplayImageDIResolver;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.graphic.image.IOpenPatricianPainter;
import ch.sahits.game.openpatrician.client.IClient;
import ch.sahits.game.rendering.AbstractRenderPart;
/**
 * This is the main in game view. The main part of this class is to delegate further down
 * and to draw the frame borders over the delegated stuff.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Sep 30, 2011
 *
 */
public class MainGameView extends AbstractRenderPart {
	/** Height of the map overview */
	private static final int MAP_OVERVIEW_HEIGHT = 207;
	/** Width of the left panels (without borders) */
	private static final int LEFT_STATUS_PANEL_WIDTH = 250;
	/** Height of the top status bar */
	private static final int TOP_STATUS_HEIGHT = 70;
	/** Width/height of the border */
	private static final int BORDER_SPACING = 10;
	/** Height of the horizontal fries */
	private static final int HORIZONTAL_DECO_HEIGHT = 20;
	/** Height of the menu bar */
	private static final int MENU_HEIGHT = 71;
	/** Height of the notice view */
	private static final int NOTICE_HEIGHT = 250;
	/** Width og the left panels inclusive borders */
	private static final int LEFT_PANEL_WIDTH = 310;
	/** The minimal height the display must have */
	public static final int MINMIMAL_DISPLAY_HEIGHT = TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+TOP_STATUS_HEIGHT+NOTICE_HEIGHT-2;
	/** View holding the sea or city overview map */
	private final MapView mapOverview;
	/** View holding the main menu */
	private final MainMenu mainMenu;
	/** View holding the sub menu (ship or city) menu */
	private final SubMenu subMenu;
	/** View holding notices and state information */
	private final NoticeView noticeView;
	/** View status holding money and time */
	private final GameStatus gameStatus;
	/** Status view of the fleet or city */
	private final ViewStatus viewStatus;
	/** Main view of the map or the city */
	private final MainView mainView;
	/**
	 * Image loader to retrieve the images
	 */
	private final IImageLoader loader;
	/** Cached background image */
	private final BufferedImage background;
//	private final IClient client;

	
	public MainGameView(Rectangle rect,IClient cl) {
		super(rect);
//		client=cl;
		if (cl==null){
			throw new NullPointerException("Client instance may not be null");
		}
		Rectangle rectangle = new Rectangle(rect.x+BORDER_SPACING, rect.y+HORIZONTAL_DECO_HEIGHT, LEFT_STATUS_PANEL_WIDTH, TOP_STATUS_HEIGHT-HORIZONTAL_DECO_HEIGHT);
		
		DisplayImageDIResolver resolver = DisplayImageDIResolver.getInstance();
		
		loader = resolver.getMainScreenLoader(); // loader from the txt file
		background = createBackground(resolver);
			
		gameStatus = new GameStatus(rectangle,cl);
		gameStatus.setEnabled(true);		
		rectangle = new Rectangle(rect.x+LEFT_PANEL_WIDTH, rect.y+HORIZONTAL_DECO_HEIGHT, rect.width-(LEFT_PANEL_WIDTH+BORDER_SPACING), TOP_STATUS_HEIGHT-HORIZONTAL_DECO_HEIGHT);
		viewStatus = new ViewStatus(rectangle,cl);
		viewStatus.setEnabled(true);
		rectangle = new Rectangle(rect.x+5, rect.y+TOP_STATUS_HEIGHT+8, 236, 192); // inner heights
		mapOverview = new MapView(rectangle);
		mapOverview.setEnabled(true);
		// Main menu in over tiled background
		rectangle = new Rectangle();
		rectangle.x=rect.x+15;
		rectangle.y=rect.y+TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+10;
		rectangle.height=48; // inner height
		rectangle.width=221; // inner width
		mainMenu = new MainMenu(rectangle);
		mainMenu.setEnabled(true);
		// Sub menu over architrave
		rectangle = new Rectangle();
		rectangle.x=rect.x+BORDER_SPACING;
		rectangle.y=rect.y+TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+HORIZONTAL_DECO_HEIGHT;
		rectangle.height=TOP_STATUS_HEIGHT-HORIZONTAL_DECO_HEIGHT;
		rectangle.width=LEFT_STATUS_PANEL_WIDTH;
		subMenu = new SubMenu(rectangle);
		subMenu.setEnabled(true);
		// notice board
		rectangle = new Rectangle();
		rectangle.x=rect.x+5;
		rectangle.y=rect.y+TOP_STATUS_HEIGHT+MAP_OVERVIEW_HEIGHT+MENU_HEIGHT+TOP_STATUS_HEIGHT+6;
		rectangle.height=238;
		rectangle.width=236;
		noticeView = new NoticeView(rectangle,loader);
		noticeView.setEnabled(true);
		rectangle = new Rectangle();
		rectangle.x=rect.x+LEFT_PANEL_WIDTH;
		rectangle.y=rect.y+TOP_STATUS_HEIGHT;
		rectangle.height=rect.height-(rectangle.y+BORDER_SPACING);
		rectangle.width=rect.width-(LEFT_PANEL_WIDTH+BORDER_SPACING);
		mainView = new MainView(rectangle,loader,cl);
		mainView.setEnabled(true);

	}


	@Override
	public void gameRender(Graphics gScr) {
		Color oldColor = gScr.getColor();
		// For testing
//		gScr.setColor(Color.ORANGE);
//		gScr.fillRect(rect.x, rect.y, rect.width, rect.height);

		gScr.drawImage(background, 0, 0, null);
		
		// render the view
		gameStatus.gameRender(gScr);
		viewStatus.gameRender(gScr);
		mapOverview.gameRender(gScr);
		mainMenu.gameRender(gScr);
		subMenu.gameRender(gScr);
		noticeView.gameRender(gScr);
		mainView.gameRender(gScr);
		gScr.setColor(oldColor);
	}


	/**
	 * Create the background image
	 * @return
	 */
	private BufferedImage createBackground(DisplayImageDIResolver resolver) {
		BufferedImage background = new BufferedImage(rect.width, rect.height, BufferedImage.TYPE_4BYTE_ABGR);
		IOpenPatricianPainter opPainter = resolver.getOpenPatricianPainter();
		Graphics gScr = background.getGraphics();
		// Render the frame
		BufferedImage img = loader.getImage("FrameLayout");
		int width = img.getWidth();
		gScr.drawImage(img, rect.x, rect.y, null);
		// Compute the top fringe
		int remWidth = rect.width-width;
		img = loader.getImage("topDeco");
		int decoWidth = img.getWidth();
		int countComplete = remWidth/decoWidth; // number the deco can be drawn complete
		int partialWidth = remWidth-(countComplete*decoWidth);
		for (int i=0;i<countComplete;i++){
			int startX = rect.x+width+(i*decoWidth);
			gScr.drawImage(img, startX, rect.y, null);
		}
		int startX = rect.x+width+(countComplete*decoWidth);
		gScr.drawImage(img, startX, rect.y, partialWidth, img.getHeight(), null);
		
		// draw right border
		final  int cornerDim = 10;
		final int mortarDim = 12;
		final int defaultPlateding = 200;
		countComplete = (int)Math.round((rect.height-TOP_STATUS_HEIGHT-1.5*cornerDim)/(mortarDim+defaultPlateding));
		// compute the scale in y direction
		int constantBuffer = cornerDim+cornerDim/2+mortarDim*countComplete-mortarDim;
		int platingDestDim =  rect.height-TOP_STATUS_HEIGHT-constantBuffer;
		img = loader.getImage("reightBorderMiddle");
		int platingOrigDim = countComplete*img.getHeight();
		double scale = platingDestDim*1.0/platingOrigDim;
		BufferedImage plate = opPainter.scaleY(img, (int)Math.rint(img.getHeight()*scale));
		startX = rect.x+rect.width-img.getWidth();
		int startY = rect.y+TOP_STATUS_HEIGHT;
		// draw top right corner
		img = loader.getImage("rightTopBorderCorner");
		img = img.getSubimage(0, img.getHeight()/2, img.getWidth(), img.getHeight()/2);
		gScr.drawImage(img, startX, startY, null);
		startY += img.getHeight();
		img = loader.getImage("reightBorderMortar");
		for (int h=0;h<countComplete;h++){
			if (h<countComplete-1){
				gScr.drawImage(plate, startX, startY, null);
				startY += plate.getHeight();
				gScr.drawImage(img, startX, startY, null);
				startY += img.getHeight();
			} else {
				// fill all the excessive space
				int height = rect.height-cornerDim-startY;
				img = loader.getImage("reightBorderMiddle"); // overriding since there is no more mortar to be drawn
				plate = opPainter.scaleY(img, height);
				gScr.drawImage(plate, startX, startY, null);
				startY += height;
			}
		}
		img = loader.getImage("rightBottomBorderCorner");
		gScr.drawImage(img, startX, startY, null);
		
		// Draw the bottom border
		countComplete = (int)Math.round((rect.width-LEFT_PANEL_WIDTH-1.5*cornerDim)/(mortarDim+defaultPlateding));
		// compute the scale in x direction
		platingDestDim =  rect.width-LEFT_PANEL_WIDTH-constantBuffer;
		img = loader.getImage("bottomBorderMiddle");
		platingOrigDim = countComplete*img.getWidth();
		scale = platingDestDim*1.0/platingOrigDim;
		plate = opPainter.scaleX(img, (int)Math.rint(img.getWidth()*scale));
		startX = rect.x+LEFT_PANEL_WIDTH;
		// startY must not be adjusted is correct from above
		// draw top right corner
		img = loader.getImage("leftBottomBorderCorner");
		img = img.getSubimage(img.getWidth()/2, 0, img.getWidth()/2, img.getHeight());
		gScr.drawImage(img, startX, startY, null);
		startX += img.getWidth();
		img = loader.getImage("bottomBorderMortar");
		for (int w=0;w<countComplete;w++){
			if (w<countComplete-1){
				gScr.drawImage(plate, startX, startY, null);
				startX += plate.getWidth();
				gScr.drawImage(img, startX, startY, null);
				startX += img.getWidth();
			} else {
				// fill all the excessive space
				width = rect.width-cornerDim-startX;
				img = loader.getImage("bottomBorderMiddle"); // overriding since there is no more mortar to be drawn
				plate = opPainter.scaleX(img, width);
				gScr.drawImage(plate, startX, startY, null);
				startX += width;
			}
		}
		// render deco under left side panel
		int height = rect.height-MINMIMAL_DISPLAY_HEIGHT-rect.y;
		String ornamentName="Ornament310x310";
		if (height<125){
			ornamentName="Ornament310x94";
		} else if (height<158){
			ornamentName="Ornament310x155";
		} else if (height<173){
			ornamentName="Ornament310x162";
		} else if (height<298){
			ornamentName="Ornament310x185";
		}
		img = loader.getImage(ornamentName);
		img = opPainter.scaleY(img, height);
		gScr.drawImage(img, rect.x, rect.y+MINMIMAL_DISPLAY_HEIGHT, null);
		
		return background;
	}
	@Override
	public void gameUpdate(Event e, Object notice) {
		// TODO Auto-generated method stub
		
	}


}
