package ch.sahits.game.graphic.display.dialog;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.Event;
import ch.sahits.game.event.IEventListener;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.display.gameplay.MainView;
import ch.sahits.game.graphic.display.model.CityPlayerProxy;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.rendering.RenderablePart;

/**
 * This is a dialog that renderes a wooden frame with an inlaid parchment at a given
 * position. Based on what the dialog represents the subclass may implement a {@link RenderablePart}.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 18, 2011
 *
 */
public abstract class OpenPatricianGameDialog {
	/**
	 * Frame that is rendered
	 */
	private final BufferedImage img;
	/** Top left corner of the frame */
	private final Rectangle bounds;
	protected Insets inset;
	protected IImageLoader loader;
	/** Reference to the city view model */
	protected final CityPlayerProxy city;
	
	
	
	public OpenPatricianGameDialog(Point topLeft, IImageLoader loader, double scale, CityPlayerProxy cityProxy) {
		super();
		init();
		this.city=cityProxy;
		initiatePolygons();
		this.loader=loader;
		initInsets(scale);
		img = initBackgroundImage(loader,topLeft);
		bounds = new Rectangle(topLeft, new Dimension(img.getWidth(), img.getHeight()));
	}
	/**
	 * Compute the insets as a factor of the scaling
	 * @param scale
	 */
	protected void initInsets(double scale) {
		int top = (int) Math.ceil(scale*30);
		int side = (int) Math.ceil(scale*20);
		inset = new Insets(top, side, top, side);
	}
	/**
	 * Initialize class specific stuff which might be need later in the initialisation
	 */
	protected void init() {
		// may be overridden
		
	}
	/**
	 * Instanciate the any polygon member variables.
	 * This method is intended to be overriden if the dialog uses polygons.
	 */
	protected void initiatePolygons() {
		// may be overridden
		
	}
	/**
	 * Initialize the background image that is stored as a reference. This method
	 * should be overridden by any subclass that contains static elements in the dialog.
	 * This implementation returns a copy of the background image stored in the cache, so that
	 * image can be further processed, without inflicting conflicts.
	 * @param loader ImageLoader to retrieve images
	 * @return Image that will be stored as background
	 */
	protected BufferedImage initBackgroundImage(IImageLoader loader,Point topLeft) {
		return deepCopy(loader.getImage("fringe"));
	}

	public final Rectangle getBounds() {
		return bounds;
	}


	public void gameRender(Graphics gScr) {
		gScr.drawImage(img, bounds.x, bounds.y, null);
	}
	/**
	 * Retrieve the insets of the dialog border
	 * @return
	 */
	public Insets getInset() {
		return inset;
	}
	/**
	 * Create a copy of an image
	 * @param bi
	 * @return
	 */
	private static BufferedImage deepCopy(BufferedImage bi) {
		 ColorModel cm = bi.getColorModel();
		 boolean isAlphaPremultiplied = cm.isAlphaPremultiplied();
		 WritableRaster raster = bi.copyData(null);
		 return new BufferedImage(cm, raster, isAlphaPremultiplied, null);
	}

	protected static class CloseAction implements Runnable{

		private final IEventListener dialog;
		
		
		public CloseAction(IEventListener dialog) {
			super();
			this.dialog = dialog;
		}


		@Override
		public void run() {
			new ViewChangeEvent(MainView.class).notify(EViewChangeEvent.CLOSE_DIALOG);
			Event.remove(dialog);
		}		
	}

}
