package ch.sahits.game.graphic.display.dialog;

import java.awt.Color;
import java.awt.FontFormatException;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.font.GlyphVector;
import java.awt.image.BufferedImage;
import java.io.IOException;

import org.apache.log4j.Logger;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.Event;
import ch.sahits.game.event.IEventListener;
import ch.sahits.game.event.KeyPressEvent;
import ch.sahits.game.event.MouseClickEvent;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.display.gameplay.MainView;
import ch.sahits.game.graphic.display.model.CityPlayerProxy;
import ch.sahits.game.graphic.display.notice.NoticeView;
import ch.sahits.game.graphic.display.util.ClickablePolygons;
import ch.sahits.game.graphic.image.DisplayImageDIResolver;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.graphic.image.IOpenPatricianPainter;
import ch.sahits.game.graphic.image.model.NamedPolygon;
import ch.sahits.game.image.IImageUtilities;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.rendering.RenderablePart;
/**
 * Trade dialog for buying/selling from/to the city involving either a ship or the branch office.
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 18, 2011
 *
 */
public class TradeDialog extends OpenPatricianGameDialog implements
		RenderablePart, IEventListener, ITransferable {
	private static final Logger logger = Logger.getLogger(TradeDialog.class);
	private boolean enabled = true;
	/** The amount that is moved with one click */
	private ETransferAmount movableAmount = ETransferAmount.ONE;
	private ClickablePolygons polygons;
	private ClickablePolygons footerPolygons;
	/** Hold all the positioning information, initialized in initBackgroundImage */
	private DialogPositions positions;
	private ETransferDialogType type;
	/** reference to the utilities for image manipulation */
	private IImageUtilities imageUtils;
	private IOpenPatricianPainter opPainter;

	public TradeDialog(Point topLeft, double scale, CityPlayerProxy city,ETransferDialogType dialogType) {
		super(topLeft, DisplayImageDIResolver.getInstance().getMainScreenLoader(),scale,city);
		type = dialogType;
		Event.add(this);
	}
	
		
	@Override
	protected void init() {
		DisplayImageDIResolver resolver = DisplayImageDIResolver.getInstance();
		opPainter = resolver.getOpenPatricianPainter();
		imageUtils = resolver.getImageUtilities();
		super.init();
	}


	@Override
	protected void initiatePolygons() {
		polygons = new ClickablePolygons();
		footerPolygons = new ClickablePolygons();
	}
	
	
	@Override
	public ETransferDialogType getDialogType() {
		return type;
	}
	
	@Override
	public void setDialogType(ETransferDialogType type) {
		synchronized (type) { // do not change the type while actions are executed that require the type
			this.type = type;
		}
	}

	@Override
	public int getAmount(int maxMount){
		switch (movableAmount){
		case ONE:
			return 1;
		case FIVE:
			if (maxMount<5) return maxMount;
			return 5;
		case MAX:
			return maxMount;
		default:
			return 1;
		}
	}
	@Override
	public ETransferAmount getMovableAmount(){
		return movableAmount;
	}
	@Override
	public void gameUpdate(Event e, Object notice) {
		if (e instanceof KeyPressEvent){
			KeyEvent key= ((KeyPressEvent)e).getKey();
			if (key.getKeyCode()==KeyEvent.VK_ESCAPE){
				new ViewChangeEvent(MainView.class).notify(EViewChangeEvent.CLOSE_DIALOG);
				new ViewChangeEvent(NoticeView.class).notify(EViewChangeEvent.NOTICE_HIDE);
				Event.remove(this);
			}
		}
		if (e instanceof MouseClickEvent){
			polygons.testAndExecute((Point) notice);
			footerPolygons.testAndExecute((Point) notice);
		}
		if (e instanceof ViewChangeEvent){
			ViewChangeEvent event = (ViewChangeEvent) e;
			if (event.getAddresse()==getClass()){
				if (notice instanceof ETransferDialogType){
					logger.debug("Change dialog type from "+type+" to "+notice);
					setDialogType((ETransferDialogType) notice);
				}
			}
		}
	}

	@Override
	public boolean isEnabled() {
		return enabled;
	}

	@Override
	public void setEnabled(boolean flag) {
		enabled=flag;
	}
	/**
	 * Draw the table header
	 * @param g2d
	 * @param y position of the table header baseline
	 * @throws FontFormatException
	 * @throws IOException
	 */
	private void drawTableHeader(Graphics2D g2d,final int y) throws FontFormatException, IOException{
		GlyphVector gv = opPainter.createGlyphVector(g2d, "Ware", 19); // TODO externalize
		int x = positions.xWare-(int)Math.rint(gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, x, y);
		String columnTitle;
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			columnTitle = "City";
			break;
		default:
			columnTitle = "Storage";
			break;
		}
		gv = opPainter.createGlyphVector(g2d, columnTitle, 19);
		g2d.drawGlyphVector(gv, positions.xCity, y);
		gv = opPainter.createGlyphVector(g2d, "Buy", 19);
		g2d.drawGlyphVector(gv, positions.xBuy, y);
		gv = opPainter.createGlyphVector(g2d, "Sell", 19);
		g2d.drawGlyphVector(gv, positions.xSell, y);
		switch (type) {
		case CITY_TO_SHIP:
		case STORAGE_TO_SHIP:
			if (city.getActiveShip() instanceof IConvoy){
				columnTitle = "Convoy";
			} else {
				columnTitle = "Ship";
			}
			break;
		default:
			columnTitle = "Storage";
		}
		gv = opPainter.createGlyphVector(g2d, columnTitle, 19);
		g2d.drawGlyphVector(gv, positions.xShip, y);
		gv = opPainter.createGlyphVector(g2d, Character.toChars(0x00D8)[0]+"-Price", 19); // TODO do better
		g2d.drawGlyphVector(gv, positions.xPrice, y);		
	}
	

	@Override
	protected BufferedImage initBackgroundImage(IImageLoader loader,Point topLeft) {
		BufferedImage bg = super.initBackgroundImage(loader,topLeft);
		Graphics2D g2d = bg.createGraphics();
		g2d.setColor(Color.BLACK);
		try {
			GlyphVector gv = opPainter.createGlyphVector(g2d, "Trade goods", 24); // TODO externalize
			double titleHeight = gv.getVisualBounds().getHeight();
			int length = (int) Math.rint(gv.getVisualBounds().getWidth());
			int x = ((bg.getWidth()-getInset().left-getInset().right)-length)/2;
			int y = getInset().top+10+(int)Math.rint(titleHeight*2/3);
			g2d.drawGlyphVector(gv, x, y);
			// Dynamic row
			y += titleHeight;
			y += (int)Math.rint(titleHeight*2/3);
			
			BufferedImage waxSeal = loader.getImage("waxseal");
			int leftBorderX = topLeft.x+getInset().left;
			int xWare = leftBorderX+100; // right aligned
			int xCity = leftBorderX+110;
			int xBuy = leftBorderX+170; // size of the image + 10
			int xSell = leftBorderX+220;// size of the image + 10
			int xShip = leftBorderX+270;
			int xPrice = leftBorderX+330;
			int yLineDiff = waxSeal.getHeight();
			// Initialize the position relative to the frame
			positions = new DialogPositions(titleHeight, yLineDiff, leftBorderX, xWare, xCity, xBuy, xSell, xShip, xPrice);
			
			leftBorderX = getInset().left; // Use local coordinates
			xWare = leftBorderX+100; // right aligned
			xCity = leftBorderX+110;
			xBuy = leftBorderX+170; // size of the image + 10
			xSell = leftBorderX+220;// size of the image + 10
			xShip = leftBorderX+270;
			xPrice = leftBorderX+330;
			
			// table header
			y += positions.lineHeight; // more space to the title
			y += positions.lineHeight;
			
			// Table
			EWare[] wares = EWare.values();
			for (EWare ware : wares) {
				// Ware name
				y += positions.lineHeight;
				gv = opPainter.createGlyphVector(g2d, ware.getLocalDisplayName(), 18);
				x = xWare-(int)Math.rint(gv.getVisualBounds().getWidth());
				g2d.drawGlyphVector(gv, x, y);
				// Amount available
				
				// Buy button
				g2d.drawImage(waxSeal, xBuy,y-(int)(positions.lineHeight*0.8), null);
				NamedPolygon polygon = new NamedPolygon(ware.name()+"Buy");
				// The positions of the polygons must be global
				polygon.addPoint(positions.xBuy, topLeft.y+y-(int)(positions.lineHeight*0.8));
				polygon.addPoint(positions.xBuy+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
				polygon.addPoint(positions.xBuy+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
				polygon.addPoint(positions.xBuy, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
				polygons.add(polygon, new BuyProxyAction(ware,this));
				
				// Sell button
				g2d.drawImage(waxSeal, xSell,y-(int)(positions.lineHeight*0.8), null);
				polygon = new NamedPolygon(ware.name()+"Sell");
				polygon.addPoint(positions.xSell, topLeft.y+y-(int)(positions.lineHeight*0.8));
				polygon.addPoint(positions.xSell+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
				polygon.addPoint(positions.xSell+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
				polygon.addPoint(positions.xSell, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
				polygons.add(polygon, new SellProxyAction(ware,this));
			}
			// Footer buttons
			y = bg.getHeight()-getInset().bottom-positions.lineHeight;
			x = getInset().left+30;
			int footerWidth = bg.getWidth()-getInset().left-getInset().right-2*30;
			// 1 item
			g2d.setColor(new Color(0xEA,0xC1,0x17)); // Gold
			NamedPolygon polygon;
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("One");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new TransferAmountChangeAction(ETransferAmount.ONE));
			gv = opPainter.createGlyphVector(g2d, "1", 18);
			int xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y);
			x += footerWidth/4;
			// 5 items
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("five");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new TransferAmountChangeAction(ETransferAmount.FIVE));
			gv = opPainter.createGlyphVector(g2d, "5", 18);
			xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y); // centeralign
			x += footerWidth/4;
			// Max item
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("Max");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new TransferAmountChangeAction(ETransferAmount.MAX));
			gv = opPainter.createGlyphVector(g2d, "Max", 18); //externalize
			xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y); // centeralign
			x += footerWidth/4;
			// close
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("Close");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new CloseAction(this));
			gv = opPainter.createGlyphVector(g2d, "X", 18);
			xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y); // centeralign


		} catch (FontFormatException e1) {
			e1.printStackTrace();
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		return bg;
	}
	@Override
	public void gameRender(Graphics gScr) {
		Color oldColor = gScr.getColor();
		super.gameRender(gScr);
		gScr.setColor(Color.BLACK);
		Graphics2D g2d = (Graphics2D) gScr;
		try {
			// Dialog title
			int y = getBounds().y+getInset().top+10+(int)Math.rint(positions.titleHeigth*2/3);
			final BufferedImage barrel = loader.getImage("barrel_icon");
			final BufferedImage bale = loader.getImage("bale_icon");
			
			// Sub title
			y += positions.titleHeigth;
			int leftBorderX = getBounds().x+getInset().left;
			int x = leftBorderX + 20;
			ICity cityModel = city.getCity();
			IShip ship = city.getActiveShip();
			ITradingOffice office = city.getPlayer().findTradingOffice(cityModel);
			y += (int)Math.rint(positions.titleHeigth*2/3);
			
			synchronized (type) { // Avoid change of the type during rendering of one screen
				drawSubTitle(g2d, y, barrel, x, cityModel, ship, office);
				BufferedImage waxSeal = loader.getImage("waxseal");
				// Table header
				y += positions.lineHeight; // more space to the title
				y += positions.lineHeight;
				drawTableHeader(g2d, y);
				// Table
				EWare[] wares = EWare.values();
				for (EWare ware : wares) {
					// Ware name
					y += positions.lineHeight;

					// Available amount in the city
					int availableAmount = drawAvailableAmount(g2d, cityModel,office,
							ware, barrel, bale, y);

					g2d.setColor(new Color(0xEA, 0xC1, 0x17)); // Gold

					// buy
					drawBuyIconText(g2d, availableAmount, ware, waxSeal, y);

					// sell
					drawSellIconText(g2d, availableAmount, ware, waxSeal, y);

					g2d.setColor(Color.BLACK);

					// amount loaded
					drawStoredAmount(g2d, ware, ship, office, y);

					// average price
					drawAVGPrice(g2d, ware, ship, office, y);
				} // end for wares
			}
			
		} catch (FontFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		gScr.setColor(oldColor);
	}
	/**
	 * Draw the average price of the wares stored
	 * @param g2d 2D graphics context used for drawing
	 * @param ware Reference to the ware
	 * @param ship Reference to the ship
	 * @param office Trading office in the city, may be null if it does not exist
	 * @param y position of from the top
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private void drawAVGPrice(Graphics2D g2d, IWare ware, IShip ship,ITradingOffice office, int y) throws FontFormatException, IOException{
		/* If storage is on the ship retrieve the AVG price of all the wares on the ship
		 * else retrieve the AVG price from the storage
		 */
		BufferedImage coin = loader.getImage("coin_icon");
		final int coinSpace = 3;
		final int coindWidth = coin.getWidth();
		final int maxPriceWidth = (int)opPainter.createGlyphVector(g2d, "00000", 18).getVisualBounds().getWidth();
		int xRight = positions.xPrice+10+maxPriceWidth+coinSpace+coindWidth;
		AmountablePrice amounable = getStoredAmount(ware, ship, office);
		int storedAmount =amounable.getAmount();
		int value = 0;
		if (storedAmount>0){
			value = amounable.getAVGPrice();
			GlyphVector gv = opPainter.createGlyphVector(g2d, String.valueOf(value), 18);
			int x = xRight-((int)gv.getVisualBounds().getWidth()+coinSpace+coindWidth);
			g2d.drawGlyphVector(gv, x, y);
			x = xRight-coindWidth;
			y -= (int) Math.ceil(gv.getVisualBounds().getHeight());
			g2d.drawImage(coin, x, y, null);
		}
	}
	/**
	 * Retrieve the amountable object for the ware that is stored. This methods honors the type of
	 * the dialog and retrieves the amount either from the ship or the trading office.
	 * @param ware for which the object is to be retrieved
	 * @param ship Reference to the ship
	 * @param office Reference to the trading office
	 * @return Amountable object for the ware
	 */
	private AmountablePrice getStoredAmount(IWare ware, IShip ship,
			ITradingOffice office) {
		AmountablePrice amounable;
		switch (type) {
		case CITY_TO_SHIP:
		case STORAGE_TO_SHIP: // stored here means stored on the ship
			amounable = ship.getWare(ware);
			break;
		case CITY_TO_STORAGE:
			amounable = office.getWare(ware);
			break;
		default:
			// This should only occur for weapons which have their own dialog
			throw new IllegalStateException("Dialog type "+type+" not handled");
		}
		return amounable;
	}
	/**
	 * Draw the stored amount (second last column). Stored refers either to ship or the storage based on
	 * the current dialog type.
	 * @param g2d 2D graphics context used for drawing
	 * @param ware Reference to the ware
	 * @param ship Reference to the ship
	 * @param y position of from the top
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private void drawStoredAmount(Graphics2D g2d, IWare ware, IShip ship, ITradingOffice office, int y) throws FontFormatException, IOException{
		AmountablePrice amounable = getStoredAmount(ware, ship, office);
		int value = amounable.getAmount(); // differ between ship and convoi
		if (value>0){
			GlyphVector gv = opPainter.createGlyphVector(g2d, String.valueOf(value), 18);
			g2d.drawGlyphVector(gv, positions.xShip, y); // this value is dynamic
			// draw barrel or burden
		}
	}
	/**
	 * Draw the text on the wax seal corresponding to the sell button
	 * @param g2d 2D graphics context used for drawing
	 * @param availableAmount amount that is actually available
	 * @param ware Reference to the ware
	 * @param waxSeal icon
	 * @param y position of from the top
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private void drawSellIconText(Graphics2D g2d, int availableAmount, IWare ware, BufferedImage waxSeal, int y) throws FontFormatException, IOException{
		int amount = getAmount(availableAmount);
		String value;
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			if (movableAmount==ETransferAmount.MAX){
				value = String.valueOf(ware.getMaxValueSell());
			} else if (amount>0){
				value = String.valueOf(ware.computeSellPrice(availableAmount, amount));
			} else {
				value = String.valueOf(ware.getMaxValueSell());
			}
			break;
		default:
			value = ">";
		}
		GlyphVector gv = opPainter.createGlyphVector(g2d, value, 18);
		int xPadding = imageUtils.computeCenterAlignX(positions.xSell, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, xPadding, y);
	}
	/**
	 * Draw the text on the wax seal corresponding to the buy button
	 * @param g2d 2D graphics context used for drawing
	 * @param availableAmount amount that is actually available
	 * @param ware Reference to the ware
	 * @param waxSeal icon
	 * @param y position of from the top
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private void drawBuyIconText(Graphics2D g2d, int availableAmount, IWare ware, BufferedImage waxSeal, int y) throws FontFormatException, IOException{
		int amount = getAmount(availableAmount);
		String value;
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			if (availableAmount>0){
				value = String.valueOf(ware.computeBuyPrice(availableAmount, amount));
			} else {
				value = "0"; // cannot buy anything if nothing is there
			}
			break;
		default:
			value = "<";
		}
		GlyphVector gv = opPainter.createGlyphVector(g2d, value, 18);
		int xPadding = imageUtils.computeCenterAlignX(positions.xBuy, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, xPadding, y);

	}
	
	/**
	 * Draw the available amount (second column). Available refers either to city or the storage based on
	 * the current dialog type.
	 * @param g2d 2D graphics context used for drawing
	 * @param city Reference to the city
	 * @param ware Reference to the ware
	 * @param barrel icon
	 * @param bale icon
	 * @param y position of from the top
	 * @return the available amount
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private int drawAvailableAmount(Graphics2D g2d, ICity city, ITradingOffice office, IWare ware,BufferedImage barrel, BufferedImage bale,int y) throws FontFormatException, IOException{
		AmountablePrice available;
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			available = city.getWare(ware);
			break;
		default:
			available = office.getWare(ware);
			break;
		}
		int availableAmount = available.getAmount();
		int value = availableAmount; // amount available
		if (value>0){
			GlyphVector gv = opPainter.createGlyphVector(g2d, String.valueOf(value), 18);
			g2d.drawGlyphVector(gv, positions.xCity, y);
			// draw barrel or burden
			if (ware.getSizeAsBarrels()==1){
				// barrel
				g2d.drawImage(barrel, (int)(positions.xCity+gv.getVisualBounds().getWidth())+5, y-positions.lineHeight*2/3, null);
			} else {
				g2d.drawImage(bale, (int)(positions.xCity+gv.getVisualBounds().getWidth())+5, y-positions.lineHeight*2/3, null);
			}
		}
		return availableAmount;

	}
	/**
	 * Draw the sub title appropriate for the current dialog type onto the graphics context
	 * @param g2d "D graphics context
	 * @param y position from the top to the sub title
	 * @param barrel Icon
	 * @param x position from the left to the sub title
	 * @param city reference to the city
	 * @param ship reference to the ship
	 * @throws FontFormatException Error while creating a font
	 * @throws IOException Error while reading a resource
	 */
	private void drawSubTitle(Graphics2D g2d, int y,
			final BufferedImage barrel, int x, ICity city, IShip ship, ITradingOffice office)
			throws FontFormatException, IOException {
		// City name
		String name;
		// FIXME this is not correct in all cases
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			name = city.getName()+" "+city.getPopulation();
			break;
		case STORAGE_TO_SHIP:{
			int loadable = ship.getSize()-ship.getUpgradeSpaceReduction();
			int stored = ship.getLoad(); // TODO subtract weapons and crew
			StringBuilder sb = new StringBuilder();
			sb.append(stored).append(" / ").append(loadable).append(" ").append(ship.getName());
			name = sb.toString();
			break;
		}
		default:
			throw new IllegalStateException("Dialog type "+type+" not handled");
		}
		GlyphVector gv = opPainter.createGlyphVector(g2d, name, 18);
		g2d.drawGlyphVector(gv, x, y);
		
		// Barrel icon
		x += (int)Math.rint(gv.getVisualBounds().getWidth())+50;
		g2d.drawImage(barrel, x, y-(int)Math.rint(positions.titleHeigth), null); // Is y correct ???
		
		// loaded on ship and ship capacity
		x += 50;
		StringBuilder sb = new StringBuilder();
		int loadable;
		int stored;
		switch (type) {
		case CITY_TO_SHIP:
			loadable = ship.getSize()-ship.getUpgradeSpaceReduction();
			stored = ship.getLoad(); // TODO subtract weapons and crew
			break;
		case CITY_TO_STORAGE:
		case STORAGE_TO_SHIP:
			loadable = office.getCapacity();
			stored = office.getStored();
			break;
		default:
			throw new IllegalStateException("Dialog type "+type+" not handled");
		}
		sb.append(stored).append(" / ").append(loadable).append(" ");
		switch (type) {
		case CITY_TO_SHIP:
			sb.append(ship.getName());
			break;
		case STORAGE_TO_SHIP:
		case CITY_TO_STORAGE:
			sb.append("Storage");
			break;
		default:
			break;
		}
		gv = opPainter.createGlyphVector(g2d, sb.toString(), 18);
		g2d.drawGlyphVector(gv, x, y);
	}
	@Override
	public CityPlayerProxy getCityPlayerProxy() {
		return city;
	}
	


	/**
	 * Action to change the chunck size that is to be moved.
	 * @author Andi Hotz, (c) Sahits GmbH, 2011
	 * Created on Nov 25, 2011
	 *
	 */
	private class TransferAmountChangeAction implements Runnable{
		
		private final ETransferAmount size;
		
		public TransferAmountChangeAction(ETransferAmount size) {
			super();
			this.size = size;
		}

		@Override
		public void run() {
			movableAmount=size;
			// Change the visual of the pressed button
		}
	}
	/**
	 * This class holds the positions where different elements are
	 * placed on the dialog.
	 * @author Andi Hotz, (c) Sahits GmbH, 2011
	 * Created on Dec 9, 2011
	 *
	 */
	private static class DialogPositions{
		private final double titleHeigth;
		private final int lineHeight;
//		private final int leftBorder;
		private final int xWare;
		private final int xCity;
		private final int xBuy;
		private final int xSell;
		private final int xShip;
		private final int xPrice;
		public DialogPositions(double titleHeigth, int lineHeight,
				int leftBorder, int xWare, int xCity, int xBuy, int xSell,
				int xShip, int xPrice) {
			super();
			this.titleHeigth = titleHeigth;
			this.lineHeight = lineHeight;
//			this.leftBorder = leftBorder;
			this.xWare = xWare;
			this.xCity = xCity;
			this.xBuy = xBuy;
			this.xSell = xSell;
			this.xShip = xShip;
			this.xPrice = xPrice;
		}
	
		
	}
}
