package ch.sahits.game.graphic.display.dialog;

import java.awt.Color;
import java.awt.FontFormatException;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.font.GlyphVector;
import java.awt.image.BufferedImage;
import java.io.IOException;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.Event;
import ch.sahits.game.event.IEventListener;
import ch.sahits.game.event.KeyPressEvent;
import ch.sahits.game.event.MouseClickEvent;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.display.gameplay.MainView;
import ch.sahits.game.graphic.display.model.CityPlayerProxy;
import ch.sahits.game.graphic.display.notice.NoticeView;
import ch.sahits.game.graphic.display.util.ClickablePolygons;
import ch.sahits.game.graphic.image.DisplayImageDIResolver;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.graphic.image.IOpenPatricianPainter;
import ch.sahits.game.graphic.image.model.NamedPolygon;
import ch.sahits.game.image.IImageUtilities;
import ch.sahits.game.openpatrician.model.building.ELevel;
import ch.sahits.game.openpatrician.model.building.IStorage;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.rendering.RenderablePart;
/**
 * Overview of the weapons stored in the trading office
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 25, 2012
 *
 */
public class TradingOfficeWareStorageDialog extends OpenPatricianGameDialog  implements IEventListener, RenderablePart{


	/** reference to the utilities for image manipulation */
	private IImageUtilities imageUtils;
	private IOpenPatricianPainter opPainter;
	private boolean enabled = true;
	private ClickablePolygons polygons;
	private ClickablePolygons footerPolygons;
	/** Hold all the positioning information, initialized in initBackgroundImage */
	private DialogPositions positions;

	public TradingOfficeWareStorageDialog(Point topLeft, double scale,
			CityPlayerProxy cityProxy) {
		super(topLeft, DisplayImageDIResolver.getInstance().getMainScreenLoader(), scale, cityProxy);
		Event.add(this);
	}
	@Override
	protected void init() {
		DisplayImageDIResolver resolver = DisplayImageDIResolver.getInstance();
		opPainter = resolver.getOpenPatricianPainter();
		imageUtils = resolver.getImageUtilities();
		super.init();
	}
	@Override
	protected void initiatePolygons() {
		polygons = new ClickablePolygons();
		footerPolygons = new ClickablePolygons();
	}
	@Override
	public boolean isEnabled() {
		return enabled;
	}

	@Override
	public void setEnabled(boolean flag) {
		enabled=flag;
	}
	@Override
	public void gameUpdate(Event e, Object notice) {
		if (e instanceof KeyPressEvent){
			KeyEvent key= ((KeyPressEvent)e).getKey();
			if (key.getKeyCode()==KeyEvent.VK_ESCAPE){
				new ViewChangeEvent(MainView.class).notify(EViewChangeEvent.CLOSE_DIALOG);
				new ViewChangeEvent(NoticeView.class).notify(EViewChangeEvent.NOTICE_HIDE);
				Event.remove(this);
			}
		}
		if (e instanceof MouseClickEvent){
			polygons.testAndExecute((Point) notice);
			footerPolygons.testAndExecute((Point) notice);
		}
	}
	/**
	 * Draw a text string at a defined position. The Y-position is passed as an argument, updated for the next row
	 * and returned. The X-coordinate is computed from the left border argument and the visual length of the text string
	 * @param g2d Graphic context
	 * @param y position
	 * @param leftBorder left inset
	 * @param text to be drawn
	 * @return updated y coordinate
	 * @throws FontFormatException
	 * @throws IOException
	 */
	private int drawRow(Graphics2D g2d, int y,int leftBorder, String text) throws FontFormatException, IOException{
		y += positions.lineHeight;
		GlyphVector gv = opPainter.createGlyphVector(g2d, text, 18);
		int x = leftBorder-(int)Math.rint(gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, x, y);
		return y;
	}
	@Override
	protected BufferedImage initBackgroundImage(IImageLoader loader,Point topLeft) {
		BufferedImage bg = super.initBackgroundImage(loader,topLeft);
		Graphics2D g2d = bg.createGraphics();
		g2d.setColor(Color.BLACK);
		try {
			GlyphVector gv = opPainter.createGlyphVector(g2d, "Storage", 24); // TODO externalize
			double titleHeight = gv.getVisualBounds().getHeight();
			int length = (int) Math.rint(gv.getVisualBounds().getWidth());
			int x = ((bg.getWidth()-getInset().left-getInset().right)-length)/2;
			int y = getInset().top+10+(int)Math.rint(titleHeight*2/3);
			g2d.drawGlyphVector(gv, x, y);
			// Dynamic row
			y += titleHeight;
			y += (int)Math.rint(titleHeight*2/3);
			
			BufferedImage waxSeal = loader.getImage("waxseal");
			final BufferedImage plus = loader.getImage("plusIcon");
			final BufferedImage minus = loader.getImage("minusIcon");
			final BufferedImage barrel = loader.getImage("barrel_icon");
			final BufferedImage bale = loader.getImage("bale_icon");
			BufferedImage coin = loader.getImage("coin_icon");
			
			int leftBorderX = topLeft.x+getInset().left;
			int firstColumn = leftBorderX+230; // right aligned
			int icons = leftBorderX+350;
			int xMinus = leftBorderX+300;
			int numbers = leftBorderX+335;
			int yLineDiff = waxSeal.getHeight();
			// Initialize the position relative to the frame
			positions = new DialogPositions(titleHeight, yLineDiff, numbers);
			
			leftBorderX = getInset().left; // Use local coordinates
			firstColumn = leftBorderX+230; // right aligned
			icons = leftBorderX+350;
			xMinus = leftBorderX+300;
			numbers = leftBorderX+335;
			
			
			y += 5*positions.lineHeight;
			
			// Table
			String s = "Required storage capacity";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(bale, icons,y-(int)(positions.lineHeight*0.8), null);
			
			s = "Available storage capacity";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(bale, icons,y-(int)(positions.lineHeight*0.8), null);

			s = "Additional storage capacity";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(bale, icons,y-(int)(positions.lineHeight*0.8), null);

			s = "Other goods";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(barrel, icons,y-(int)(positions.lineHeight*0.8), null);

			s = "Costs (per day)";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(coin, icons,y-(int)(positions.lineHeight*0.8), null);
			
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.lineHeight;

			s = "Guards";
			gv = opPainter.createGlyphVector(g2d, s, 18);
			length = (int) Math.rint(gv.getVisualBounds().getWidth());
			x = ((bg.getWidth()-getInset().left-getInset().right)-length)/2;
			g2d.drawGlyphVector(gv, x, y);
			
			s = "Number of Guards";
			y = drawRow(g2d, y, firstColumn, s);			
			
			x = xMinus;
			g2d.drawImage(minus, x,y-(int)(positions.lineHeight*0.8), null);
			NamedPolygon polygon = new NamedPolygon("MinusOneGuard");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+minus.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+minus.getHeight());
			polygons.add(polygon, new GuardNumberChange(-1));

			x = icons;
			g2d.drawImage(plus, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("PlusOneGuard");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+plus.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+plus.getHeight());
			polygons.add(polygon, new GuardNumberChange(1));
			
			s = "Costs (per day)";
			y = drawRow(g2d, y, firstColumn, s);
			g2d.drawImage(coin, icons,y-(int)(positions.lineHeight*0.8), null);
			
			s = "Security";
			y = drawRow(g2d, y, firstColumn, s);

			// Footer buttons
			y = bg.getHeight()-getInset().bottom-positions.lineHeight;
			x = getInset().left+30;
			int footerWidth = bg.getWidth()-getInset().left-getInset().right-2*30;
			x += 3*footerWidth/4;
			// close
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			polygon = new NamedPolygon("Close");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new CloseAction(this));
			g2d.setColor(new Color(0xEA,0xC1,0x17)); // Gold
			gv = opPainter.createGlyphVector(g2d, "X", 18);
			int xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y); // centeralign


		} catch (FontFormatException e1) {
			e1.printStackTrace();
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		return bg;
	}
	@Override
	public void gameRender(Graphics gScr) {
		Color oldColor = gScr.getColor();
		super.gameRender(gScr);
		gScr.setColor(Color.BLACK);
		Graphics2D g2d = (Graphics2D) gScr;
		try {
			// Dialog title
			int y = getBounds().y+getInset().top+10+(int)Math.rint(positions.titleHeigth*2/3);
			// Sub title
			y += positions.titleHeigth;
			y += 5*positions.lineHeight;
			ICity cityModel = city.getCity();
			y += (int)Math.rint(positions.titleHeigth*2/3);
			
			
			ITradingOffice office = city.getPlayer().findTradingOffice(cityModel);
			IStorage storage = office.getStorage();
			
			drawRow(g2d, y, positions.xNumbers, String.valueOf(office.getStored()/10));
			y += positions.lineHeight;

			drawRow(g2d, y, positions.xNumbers, String.valueOf(office.getCapacity()/10));
			y += positions.lineHeight;
			
			int additional = office.getStored()-office.getCapacity();
			if (additional<0) additional = 0;
			drawRow(g2d, y, positions.xNumbers, String.valueOf(additional/10));
			y += positions.lineHeight;
			
			drawRow(g2d, y, positions.xNumbers, String.valueOf(storage.getRentOutStorage()));
			y += positions.lineHeight;

			drawRow(g2d, y, positions.xNumbers, String.valueOf(storage.getCostsPerDay()));
			y += positions.lineHeight;
			y += positions.lineHeight;

			y += positions.lineHeight;
			y += positions.lineHeight;
			
			drawRow(g2d, y, positions.xNumbers, String.valueOf(storage.getNumberGuards()));
			y += positions.lineHeight;

			drawRow(g2d, y, positions.xNumbers, String.valueOf(storage.getGuardCostsPerDay()));
			y += positions.lineHeight;
			
			ELevel security = storage.getSecurityLevel(); // TODO externalize
			drawRow(g2d, y, positions.xNumbers, security.name());

		} catch (FontFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		gScr.setColor(oldColor);
	}


	/**
	 * This class holds the positions where different elements are
	 * placed on the dialog.
	 * @author Andi Hotz, (c) Sahits GmbH, 2011
	 * Created on Dec 9, 2011
	 *
	 */
	private static class DialogPositions{
		private final double titleHeigth;
		private final int lineHeight;
		private final int xNumbers;
		public DialogPositions(double titleHeigth, int lineHeight, int xGuards) {
			super();
			this.titleHeigth = titleHeigth;
			this.lineHeight = lineHeight;
			this.xNumbers = xGuards;
		}
		
	}
	private class GuardNumberChange implements Runnable {
		
		private final int guardNumberChange;
		

		public GuardNumberChange(int guardNumberChange) {
			super();
			this.guardNumberChange = guardNumberChange;
		}


		@Override
		public void run() {
			IStorage storage = city.getPlayer().findTradingOffice(city.getCity()).getStorage();
			storage.updateGuardsNumber(guardNumberChange);
		}

	}


}
