package ch.sahits.game.graphic.display.dialog;

import java.awt.Color;
import java.awt.FontFormatException;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.event.KeyEvent;
import java.awt.font.GlyphVector;
import java.awt.image.BufferedImage;
import java.io.IOException;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.Event;
import ch.sahits.game.event.IEventListener;
import ch.sahits.game.event.KeyPressEvent;
import ch.sahits.game.event.MouseClickEvent;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.display.gameplay.MainView;
import ch.sahits.game.graphic.display.model.CityPlayerProxy;
import ch.sahits.game.graphic.display.notice.NoticeView;
import ch.sahits.game.graphic.display.util.ClickablePolygons;
import ch.sahits.game.graphic.image.DisplayImageDIResolver;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.graphic.image.IOpenPatricianPainter;
import ch.sahits.game.graphic.image.model.NamedPolygon;
import ch.sahits.game.image.IImageUtilities;
import ch.sahits.game.openpatrician.model.building.IWeaponStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IWeapon;
import ch.sahits.game.rendering.RenderablePart;
/**
 * Overview of the weapons stored in the trading office
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 25, 2012
 *
 */
public class TradingOfficeWeaponStorageDialog extends OpenPatricianGameDialog  implements IEventListener, RenderablePart{
	/** reference to the utilities for image manipulation */
	private IImageUtilities imageUtils;
	private IOpenPatricianPainter opPainter;
	private boolean enabled = true;
	private ClickablePolygons footerPolygons;
	/** Hold all the positioning information, initialized in initBackgroundImage */
	private DialogPositions positions;

	public TradingOfficeWeaponStorageDialog(Point topLeft, double scale,
			CityPlayerProxy cityProxy) {
		super(topLeft, DisplayImageDIResolver.getInstance().getMainScreenLoader(), scale, cityProxy);
		Event.add(this);
	}
	@Override
	protected void init() {
		DisplayImageDIResolver resolver = DisplayImageDIResolver.getInstance();
		opPainter = resolver.getOpenPatricianPainter();
		imageUtils = resolver.getImageUtilities();
		super.init();
	}
	@Override
	protected void initiatePolygons() {
		footerPolygons = new ClickablePolygons();
	}
	@Override
	public boolean isEnabled() {
		return enabled;
	}

	@Override
	public void setEnabled(boolean flag) {
		enabled=flag;
	}
	@Override
	public void gameUpdate(Event e, Object notice) {
		if (e instanceof KeyPressEvent){
			KeyEvent key= ((KeyPressEvent)e).getKey();
			if (key.getKeyCode()==KeyEvent.VK_ESCAPE){
				new ViewChangeEvent(MainView.class).notify(EViewChangeEvent.CLOSE_DIALOG);
				new ViewChangeEvent(NoticeView.class).notify(EViewChangeEvent.NOTICE_HIDE);
				Event.remove(this);
			}
		}
		if (e instanceof MouseClickEvent){
			footerPolygons.testAndExecute((Point) notice);
		}
	}
	/**
	 * Draw the table header
	 * @param g2d
	 * @param y position of the table header baseline
	 * @throws FontFormatException
	 * @throws IOException
	 */
	private void drawTableHeader(Graphics2D g2d,final int y) throws FontFormatException, IOException{
		GlyphVector gv = opPainter.createGlyphVector(g2d, "Weapons", 19); // TODO externalize
		int x = positions.firstColumn-(int)Math.rint(gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, x, y);
		String columnTitle = "In Stock";
		gv = opPainter.createGlyphVector(g2d, columnTitle, 19);
		g2d.drawGlyphVector(gv, positions.secondcolumn, y);
	}
	/**
	 * Draw a text string at a defined position. The Y-position is passed as an argument, updated for the next row
	 * and returned. The X-coordinate is computed from the left border argument and the visual length of the text string
	 * @param g2d Graphic context
	 * @param y position
	 * @param leftBorder left inset
	 * @param text to be drawn
	 * @return updated y coordinate
	 * @throws FontFormatException
	 * @throws IOException
	 */
	private int drawRow(Graphics2D g2d, int y,int leftBorder, String text) throws FontFormatException, IOException{
		y += positions.lineHeight;
		GlyphVector gv = opPainter.createGlyphVector(g2d, text, 18);
		int x = leftBorder-(int)Math.rint(gv.getVisualBounds().getWidth());
		g2d.drawGlyphVector(gv, x, y);
		return y;
	}
	@Override
	protected BufferedImage initBackgroundImage(IImageLoader loader,Point topLeft) {
		BufferedImage bg = super.initBackgroundImage(loader,topLeft);
		Graphics2D g2d = bg.createGraphics();
		g2d.setColor(Color.BLACK);
		try {
			GlyphVector gv = opPainter.createGlyphVector(g2d, "Weapon Inventory", 24); // TODO externalize
			double titleHeight = gv.getVisualBounds().getHeight();
			int length = (int) Math.rint(gv.getVisualBounds().getWidth());
			int x = ((bg.getWidth()-getInset().left-getInset().right)-length)/2;
			int y = getInset().top+10+(int)Math.rint(titleHeight*2/3);
			g2d.drawGlyphVector(gv, x, y);
			// Dynamic row
			y += titleHeight;
			y += (int)Math.rint(titleHeight*2/3);
			
			BufferedImage waxSeal = loader.getImage("waxseal");
			int leftBorderX = topLeft.x+getInset().left;
			int firstColumn = leftBorderX+150; // right aligned
			int secondcolumn = leftBorderX+200;
			int yLineDiff = waxSeal.getHeight();
			// Initialize the position relative to the frame
			positions = new DialogPositions(titleHeight, yLineDiff, firstColumn,secondcolumn);
			
			leftBorderX = getInset().left; // Use local coordinates
			firstColumn = leftBorderX+150; // right aligned
			
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.titleHeigth;
			// table header
			y += positions.lineHeight; // more space to the title
			y += positions.lineHeight;
			
			// Table
			for (IWeapon weapon : EWeapon.values()){
				y = drawRow(g2d, y, firstColumn, weapon.getLocalDisplayName());
			}

			// Footer buttons
			y = bg.getHeight()-getInset().bottom-positions.lineHeight;
			x = getInset().left+30;
			int footerWidth = bg.getWidth()-getInset().left-getInset().right-2*30;
			x += 3*footerWidth/4;
			// close
			g2d.drawImage(waxSeal, x,y-(int)(positions.lineHeight*0.8), null);
			NamedPolygon polygon = new NamedPolygon("Close");
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8));
			polygon.addPoint(topLeft.x+x+waxSeal.getWidth(), topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			polygon.addPoint(topLeft.x+x, topLeft.y+y-(int)(positions.lineHeight*0.8)+waxSeal.getHeight());
			footerPolygons.add(polygon, new CloseAction(this));
			g2d.setColor(new Color(0xEA,0xC1,0x17)); // Gold
			gv = opPainter.createGlyphVector(g2d, "X", 18);
			int xPadding = imageUtils.computeCenterAlignX(x, waxSeal.getWidth(), (int)gv.getVisualBounds().getWidth());
			g2d.drawGlyphVector(gv, xPadding, y); // centeralign


		} catch (FontFormatException e1) {
			e1.printStackTrace();
		} catch (IOException e1) {
			e1.printStackTrace();
		}
		return bg;
	}
	@Override
	public void gameRender(Graphics gScr) {
		Color oldColor = gScr.getColor();
		super.gameRender(gScr);
		gScr.setColor(Color.BLACK);
		Graphics2D g2d = (Graphics2D) gScr;
		try {
			// Dialog title
			int y = getBounds().y+getInset().top+10+(int)Math.rint(positions.titleHeigth*2/3);
			
			// Sub title
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.lineHeight;
			y += positions.titleHeigth;
			y += positions.lineHeight;
			ICity cityModel = city.getCity();
			y += (int)Math.rint(positions.titleHeigth*2/3);
			
			// Table header
			y += positions.lineHeight; // more space to the title
			y += positions.lineHeight;
			drawTableHeader(g2d, y);
			
			// TODO fill in the data
			IWeaponStorage storage = city.getPlayer().findTradingOffice(cityModel).getWeaponStorage();
			for (IWeapon weapon : EWeapon.values()) {
				int amount = storage.getWeapon(weapon);
				if (amount>0){
					y = drawRow(g2d, y, positions.secondcolumn+50, String.valueOf(amount));
				} else {
					y += positions.lineHeight;
				}
			}
		} catch (FontFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		gScr.setColor(oldColor);
	}


	/**
	 * This class holds the positions where different elements are
	 * placed on the dialog.
	 * @author Andi Hotz, (c) Sahits GmbH, 2011
	 * Created on Dec 9, 2011
	 *
	 */
	private static class DialogPositions{
		private final double titleHeigth;
		private final int lineHeight;
		private final int firstColumn;
		private final int secondcolumn;
		public DialogPositions(double titleHeigth, int lineHeight,
				int firstColumn,int secondcolumn) {
			super();
			this.titleHeigth = titleHeigth;
			this.lineHeight = lineHeight;
			this.firstColumn = firstColumn;
			this.secondcolumn = secondcolumn;
		}
		
	}

}
