package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.LoanerEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 06, 2015
 */
@Prototype
public class LoanerPaybackDialog extends CloseButtonDialog {
    private final int BUTTON_X = (WIDTH - 124) / 2;
    private final int BUTTON_Y = CLOSE_BTN_Y_POS - 24;
    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";


    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private LoanerEngine loanerEngine;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private Date date;
    @Autowired
    private ClientViewState viewState;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;
    private final Font font;

    private ObjectProperty<IDebt> selectedDebt = new SimpleObjectProperty<>(null);
    private VBox box;

    public LoanerPaybackDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
    }
    @PostConstruct
    private void initializeDialog() {
        final ICity city1 = city.getCity();
        loaner = (LoanerState) loanerEngine.getLoaner(city1);
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.repayLoan", new Object[]{}, locale.getCurrentLocal()));
        List<IDebt> debts = loanerEngine.findDebts(loaner, city.getPlayer());
        box = new VBox(5);
        box.setLayoutX(2 * FRAME_BORDER);
        box.setLayoutY(200);
        getContent().add(box);
        if (debts.isEmpty()) {
            String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerPaybackDialog.debtFree", new Object[]{}, locale.getCurrentLocal());
            Text subHeading = new Text();
            subHeading.setText(header);
            subHeading.setFont(font);
            box.getChildren().add(subHeading);
        } else {
            String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerPaybackDialog.subHeader", new Object[]{}, locale.getCurrentLocal());
            Text subHeading = new Text();
            subHeading.setText(header);
            box.getChildren().add(subHeading);
            for (final IDebt debt : debts) {
                int payback = calculatePaybacksum(debt);
                String cityName = city1.getName();
                final DateTime dueDate1 = debt.getDueDate();
                String dueDate = date.todisplayString(dueDate1.getDayOfMonth(), dueDate1.getMonthOfYear(), dueDate1.getYear());
                String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerPaybackDialog.payback", new Object[]{payback, cityName, dueDate}, locale.getCurrentLocal());
                final DecoratedText entry = textFactory.createDecoratedText(s, new HashMap<String, Object>());
                entry.setOnMouseReleased(new EventHandler<MouseEvent>() {
                    @Override
                    public void handle(MouseEvent mouseEvent) {
                        selectedDebt.setValue(debt);
                        deactivateAll();
                        entry.applyStyle(SELECTED_STYLE);
                    }
                });
                entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
                box.getChildren().add(entry);
            } // end for debts
            final OpenPatricianLargeWaxButton acceptBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerPaybackDialog.paybackBtn", new Object[]{}, locale.getCurrentLocal()), font);
            acceptBtn.setLayoutX(BUTTON_X);
            acceptBtn.setLayoutY(BUTTON_Y);
            acceptBtn.setDisable(true);
            selectedDebt.addListener(new ChangeListener<IDebt>() {
                @Override
                public void changed(ObservableValue<? extends IDebt> observableValue, IDebt oldDebt, IDebt newDebt) {
                    if (newDebt != null) {
                        acceptBtn.setDisable(false);
                    }
                }
            });
            acceptBtn.setOnAction(createAcceptHandler());
            getContent().addAll(acceptBtn);

        } // end else there are debts
    }
    private void deactivateAll() {
        for (Node node : box.getChildren()) {
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }
    private int calculatePaybacksum(IDebt debt) {
        // todo: andi 1/6/15: If the date is before the due date the payback sum should be reduced.
        return (int)(debt.getInterest() * debt.getAmount());
    }

    private EventHandler<MouseEvent> createAcceptHandler() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                final ICompany company = city.getPlayer().getCompany();
                long cash = company.getCash();
                final IDebt debt = selectedDebt.get();
                int debtTotal = calculatePaybacksum(debt);
                if (cash < debtTotal) {
                    getContent().remove(box);
                    box = new VBox(5);
                    box.setLayoutX(2 * FRAME_BORDER);
                    box.setLayoutY(200);
                    getContent().add(getContent().size() - 2, box);
                    String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerPaybackDialog.noRepay", new Object[]{}, locale.getCurrentLocal());
                    Text subHeading = new Text();
                    subHeading.setText(header);
                    box.getChildren().add(subHeading);
                } else {
                    company.updateCash(-debtTotal);
                    loaner.getDebts().remove(debt);
                    executeOnCloseButtonClicked();
                }
            }
        };

    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
