package ch.sahits.game.graphic.display.notice;

import ch.sahits.game.event.handler.impl.EventHandlerFactory;
import ch.sahits.game.javafx.control.ScrollPaneContent;
import ch.sahits.game.javafx.util.INoticeBoardDestinction;
import ch.sahits.game.javafx.util.INoticeBoardFactory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.javafx.util.IJavaFXApplicationThreadExecution;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.collect.Lists;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.List;
import java.util.Map;

/**
 * Implementation of the notice board factory
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Nov 3, 2013
 *
 */
public class NoticeBoardFactory implements INoticeBoardFactory {
	static final Logger logger = Logger.getLogger(NoticeBoardFactory.class);
    @Autowired
    private EventHandlerFactory eventHandlerFactory;
	@Autowired
	private Locale locale;
	@Autowired
	private MessageSource messageSource;
	@Autowired
//	@Qualifier("javaFXApplicationThreadExecution")
	private IJavaFXApplicationThreadExecution threadExecutor;


	@Override
	public void populateNoticeBoardContent(ScrollPaneContent content,
			INoticeBoardDestinction destinction, ICityPlayerProxyJFX proxy, Font font) {
		if (destinction instanceof EPortSceneNoticeBoard) {
			switch ((EPortSceneNoticeBoard)destinction) {
			case TRADING:
				fillTradingContent(content, proxy, font);
				break;
			case TRADING_OFFICE:
				fillTradingOfficeContent(content, proxy, font);
				break;
			default:
				throw new RuntimeException("Not implemented: "+destinction);
			}
		} else if (destinction instanceof EMarketPlaceNoticeBoard) {
			switch ((EMarketPlaceNoticeBoard)destinction) {
			case MARKET_BOOTH:
				fillMarketPlaceBoothContent(content, proxy, font);
				break;

			default:
				throw new RuntimeException("Not implemented: "+destinction);
			}
		} else if (destinction instanceof EShipyardNoticeBoard) {
			switch ((EShipyardNoticeBoard)destinction) {
			case SHIPYARD:
				fillShipYardContent(content, proxy, font);
				break;

			default:
				throw new RuntimeException("Not implemented: "+destinction);
			}
		} else if (destinction instanceof ETavernNoticeBoard) {
			switch ((ETavernNoticeBoard)destinction) {
			case TAVERN:
				fillTavernContent(content, proxy, font);
				break;

			default:
				throw new RuntimeException("Not implemented: "+destinction);
			}
		} else if (destinction instanceof ELoanerSceneNoticeBoard) {
			switch ((ELoanerSceneNoticeBoard)destinction) {
				case LOANER:
					fillLoanerContent(content, proxy, font);
					break;
				default:
					throw new RuntimeException("Not implemented: "+destinction);
			}
		}

		else {
			throw new RuntimeException("Not yet implemented destinction type: "+destinction.getClass());
		}

	}

	private void fillLoanerContent(ScrollPaneContent content, ICityPlayerProxyJFX proxy, Font font) {
	   LoanerNoticeMetaData metaData = new LoanerNoticeMetaData(proxy);
		initializeNoticeBoard(content, metaData);
	}




	private void fillTavernContent(ScrollPaneContent content,
			ICityPlayerProxyJFX proxy, Font font) {
		TavernNoticeMetaDataJFX metaData = new TavernNoticeMetaDataJFX(proxy);
		initializeNoticeBoard(content, metaData);
		
	}


	private void fillShipYardContent(ScrollPaneContent content, ICityPlayerProxyJFX proxy, Font font) {
		ShipYardMetaDataJFX metaData = new ShipYardMetaDataJFX(proxy);
		initializeNoticeBoard(content, metaData);
	}


	private void fillMarketPlaceBoothContent(ScrollPaneContent content,
			ICityPlayerProxyJFX proxy, Font font) {
		MarketBoothDialogMetaDataJFX metaData = new MarketBoothDialogMetaDataJFX(proxy, EStringSelectionState.ACTIVE, EStringSelectionState.INACTIVE);
		initializeNoticeBoard(content, metaData);
	}


	private void fillTradingOfficeContent(ScrollPaneContent content,
			ICityPlayerProxyJFX proxy, Font font) {
		EStringSelectionState inact = EStringSelectionState.INACTIVE;
		TradingOfficeDialogMataDataJFX metaData = new TradingOfficeDialogMataDataJFX(proxy, EStringSelectionState.ACTIVE, inact, inact, inact, inact, inact);
		initializeNoticeBoard(content, metaData);
	}


	private void fillTradingContent(ScrollPaneContent content, ICityPlayerProxyJFX proxy, Font font) {
		ICity city = proxy.getCity();
		ITradingOffice office = proxy.getPlayer().findTradingOffice(city);
		IShip ship = proxy.getActiveShip();
		boolean city2ShipFlag = proxy.getActiveShip()!=null; // if true this is active
		boolean hasTradingOffice = office!=null;
		boolean hasWeapons = (city2ShipFlag&&hasTradingOffice&&office.hasWeapons()&&ship.getUpgradeSpaceReduction()>0&&ship.hasWeapons());
		EStringSelectionState city2ship;
		if (city2ShipFlag){
			city2ship = EStringSelectionState.ACTIVE;
		} else if (proxy.getPlayersShips().isEmpty()){
			// no active ship => disable
			city2ship = EStringSelectionState.DISABLE;
		} else {
			// there are ships but none is active
			city2ship = EStringSelectionState.INACTIVE;
		}
		EStringSelectionState city2office;
		if (hasTradingOffice && city2ShipFlag){
			// traiding office and active ship
			city2office=EStringSelectionState.INACTIVE;
		} else {
			// no trading office or no active ships
			city2office=EStringSelectionState.DISABLE;
		}

		EStringSelectionState office2ship;
		if (hasTradingOffice && !proxy.getPlayersShips().isEmpty() && !city2ShipFlag){
			// trading office and ships but none active
			office2ship = EStringSelectionState.ACTIVE;
		} else if (hasTradingOffice && city2ShipFlag){
			// traiding office and active ship
			office2ship=EStringSelectionState.INACTIVE;
		} else {
			// no trading office or no ships
			office2ship=EStringSelectionState.DISABLE;
		}
		EStringSelectionState weapons;
		if (hasWeapons && city2ShipFlag){
			// weapons and active ship
			weapons=EStringSelectionState.INACTIVE;
		} else {
			// no trading office or no active ships
			weapons=EStringSelectionState.DISABLE;
		}
		TradeDialogMetaDataJFX metaData = new TradeDialogMetaDataJFX(city2ship, office2ship, city2office, weapons, proxy, font);
		initializeNoticeBoard(content, metaData);
	}
	/**
	 * Initialize the notice board for trading at the harbor.
	 * @param content
	 * @param metaData
	 */
	private void initializeNoticeBoard(final ScrollPaneContent content,
			final TradeDialogMetaDataJFX metaData) {
		threadExecutor.execute(new Runnable() {
            @Override
            public void run() {
                String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.city2ship", new Object[]{}, locale.getCurrentLocal());
				if (metaData.getCityProxy().getActiveShip() instanceof IConvoy){
                    s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.city2convoi", new Object[]{}, locale.getCurrentLocal());
				}
                content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.PORT_CITY_TO_SHIP)), false, true);
                s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.ship2traidingOffice", new Object[]{}, locale.getCurrentLocal());
				if (((TradeDialogMetaDataJFX)metaData).getCityProxy().getActiveShip() instanceof IConvoy){
                    s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.convoy2tradingOffice", new Object[]{}, locale.getCurrentLocal());
				}
                content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.PORT_STORAGE_TO_SHIP)), false, false);
                s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.city2tradingOffice", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.PORT_CITY_TO_STORAGE)), false, false);
                s = messageSource.getMessage("ch.sahits.game.openpatrician.model.people.impl.WeaponsDealerState.noticeboardTitle", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.PORT_WEAPON)), true, false);
            }
        });

	}

	/**
	 * Initialite the notice board for the loaner
	 * @param content
	 * @param metaData
	 */
	private void initializeNoticeBoard(final ScrollPaneContent content, LoanerNoticeMetaData metaData) {
		threadExecutor.execute(new Runnable() {
			@Override
			public void run() {
				String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.takeLoan", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.LOANER_TAKE)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.giveLoan", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.LOANER_GIVE)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.repayLoan", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.LOANER_REPAY)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.listLoans", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.LOANER_LIST)), false, false);
			}
		});
	}
	/**
	 * Initialize the notice board for the trading office dialog.
	 * @param content that should be populated
	 * @param metaData
	 */
	private void initializeNoticeBoard(final ScrollPaneContent content,
			final TradingOfficeDialogMataDataJFX metaData) {
		threadExecutor.execute(new Runnable() {
			@Override
			public void run() {
				String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.balanceSheet", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_BALANCE)), false, true);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.personal", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_PERSONAL)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.consumtionProduction", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_CONSUMPTION_PRODUCTION)), false, false);
				s = messageSource.getMessage("ch.sahits.game.openpatrician.model.people.impl.WeaponsDealerState.noticeboardTitle", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_WEAPONS)), true, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.warehouses", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_WAREHOUSES)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.storageManager", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.TO_OFFICE_TRADING)), false, false);
			}
		});
	}
	private void initializeNoticeBoard(final ScrollPaneContent content,
			final MarketBoothDialogMetaDataJFX metaData) {
		threadExecutor.execute(new Runnable() {
			@Override
			public void run() {
				String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.stockPrices", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.MB_STOCK_PRICES)), false, true);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.consumtionProduction", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.MB_CONSUMPTION_PRODUCTION)), false, false);
			}
		});
	}
	private void initializeNoticeBoard(final ScrollPaneContent content,
			final ShipYardMetaDataJFX metaData) {
		threadExecutor.execute(new Runnable() {
			@Override
			public void run() {
				String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.construction", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_CONSTRUCTION)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipRepairDialogV2.repair", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_REPAIR)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.constructionList", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_CONSTRUCTION_LIST)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.repairList", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_REPAIR_LIST)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.upgradeShip", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_UPGRADE)), false, false);
				s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.renameShip", new Object[]{}, locale.getCurrentLocal());
				content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getViewChangeActionRunnable(ENoticeItem.SHIPYARD_RENAME)), false, false);
			}
		});
	}
	private void initializeNoticeBoard(final ScrollPaneContent content,
			final TavernNoticeMetaDataJFX metaData) {
		threadExecutor.execute(new Runnable() {
			@Override
			public void run() {
				final TavernState tavernState = metaData.tavernState;
				// todo: andi 20/12/13: make these lists observable to change when the person leaves or enters
				List<IPerson> present = tavernState.presentPersonsProperty();
				List<ISideRoomPerson> sideRoom = Lists.newArrayList();
				List<IPerson> mainRoom = Lists.newArrayList();
				IPlayer player = metaData.getCityProxy().getPlayer();
				for (IPerson pers : present) {
					if (pers instanceof ISideRoomPerson) {
						if (!tavernState.isTalkingToOtherPlayer((ISideRoomPerson) pers, player)) {
							sideRoom.add((ISideRoomPerson) pers);
						}
					} else {
						mainRoom.add(pers);
					}
				}
				for (IPerson pers : mainRoom) {
					String s = messageSource.getMessage(pers.getClass().getName() + ".noticeboardTitle", new Object[0], locale.getCurrentLocal());
					boolean active = false;
					if (tavernState.talkingToProperty().get().containsKey(pers)) {
						if (tavernState.talkingToProperty().get().get(pers).equals(player)) {
							active = true;
						}
					}
					content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getTavernChangeActionHandler(pers, tavernState, player)), false, active);
				}
				if (!sideRoom.isEmpty()) {
					String s = messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.sideRoom", new Object[]{}, locale.getCurrentLocal());
					boolean active = false;
					for (Map.Entry<IPerson, IPlayer> entry : tavernState.talkingToProperty().get().entrySet()) {
						if (entry.getKey() instanceof ISideRoomPerson
								&& entry.getValue().equals(player)) {
							// the player is talking to a person in the side room.
							active = true;
							break;
						}
					}
					content.addEntry(s, createEventHandlerFromRunnable(eventHandlerFactory.getTavernTalkToSidreroomPersonAction(sideRoom.get(0), tavernState, player)), false, active);
				}
			}
		});

	}

	
	/**
	 * Helper method to convert the old event triggering into JavaFX events.
	 * @param run
	 * @return
	 */
	private EventHandler<MouseEvent> createEventHandlerFromRunnable(final Runnable run) {
		return new EventHandler<MouseEvent>() {
			@Override
			public void handle(MouseEvent arg0) {
				run.run();
			}
		};
		
	}
	
	

}
