package ch.sahits.game.graphic.display.notice;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.NoticeBoard;
import ch.sahits.game.javafx.util.INoticeBoardDestinction;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.event.PersonLeavesTavernEvent;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;

/**
 * Actual implementation of the notice board. The updating of the board
 * happens through the client side asynchronous event bus.
 */
public class OpenPatricianNoticeBoard extends NoticeBoard {
	static final Logger logger = Logger.getLogger(OpenPatricianNoticeBoard.class);
    private ICityPlayerProxyJFX lastProxy;
    private INoticeBoardDestinction lastDestinction;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @PostConstruct
    private void initializeEventRegistration() {
        clientServerEventBus.register(this);
        clientEventBus.register(this);
    }

    /**
     * Event handler for updating the notice board
     * @param event object
     */
    @Subscribe
    public void handleNoticeBoardUpdate(NoticeBoardUpdate event) {
        ViewChangeCityPlayerProxyJFX proxy = (ViewChangeCityPlayerProxyJFX) event.getProxy();
        lastProxy = proxy;
        updateNoticeBoard(proxy);
    }

    private void updateNoticeBoard(ViewChangeCityPlayerProxyJFX proxy) {
        INoticeBoardDestinction destinction = map((EViewChangeEvent) proxy.getViewChangeEvent());
        lastDestinction = destinction;
        if (destinction != null) {
            contentContainer.clearContents();
            noticeBoardFactory.populateNoticeBoardContent(contentContainer, destinction, proxy, getFont());
        } else {
            logger.warn(proxy.getViewChangeEvent()+" is not a supported notice board change event");
        }
    }

    /**
     * Update the notice board when a person leaves
     * @param event
     */
    @Subscribe
    public void handlePersonLeaves(PersonLeavesTavernEvent event) {
        // last proxy might be null if the tavern dialog is not open, which is correct
        if (lastProxy != null && lastProxy instanceof  ViewChangeCityPlayerProxyJFX)  {
            updateNoticeBoard((ViewChangeCityPlayerProxyJFX) lastProxy);
        }
    }

    /**
     * Event handler for removing the content of the notice board.
     * @param event object
     */
    @Subscribe
    public void handleCloseNoticeBoard(NoticeBoardClose event) {
        Platform.runLater(new Runnable() {
            @Override
            public void run() {
                contentContainer.clearContents();
            }
        });
        lastProxy = null;
        lastDestinction = null;
    }

    /**
     * Handle the day change event to update the notice board.
     * @param event
     */
    @Subscribe
    public void handleDayChange(ClockTickDayChange event) {
        if (lastProxy != null && lastDestinction == ETavernNoticeBoard.TAVERN) {
            Platform.runLater(new Runnable() {
                @Override
                public void run() {
                    contentContainer.clearContents();
                    noticeBoardFactory.populateNoticeBoardContent(contentContainer, lastDestinction, lastProxy, getFont());
                }
            });
        }
    }
	/**
	 * Map the view change event to a notice board destinction. If the view change does not
	 * result in a view change null will be returned.
	 * @param viewChange
	 * @return
	 */
	private INoticeBoardDestinction map(EViewChangeEvent viewChange) {
		switch (viewChange) {
		case NOTICE_MARKET_BOOTH:
			return EMarketPlaceNoticeBoard.MARKET_BOOTH;
		case NOTICE_SHIPYARD:
			return EShipyardNoticeBoard.SHIPYARD;
		case NOTICE_TAVERN:
			return ETavernNoticeBoard.TAVERN;
		case NOTICE_TRADE:
			return EPortSceneNoticeBoard.TRADING;
		case NOTICE_TRADING_OFFICE:
			return EPortSceneNoticeBoard.TRADING_OFFICE;
        case NOTICE_LOANER:
            return ELoanerSceneNoticeBoard.LOANER;
		default:
			return null;
		}
	}
}
