package ch.sahits.game.graphic.display.scene;

import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.graphic.loaders.IPolygonLoader;
import ch.sahits.game.javafx.control.PaginationV2;
import ch.sahits.game.javafx.control.PaginationV2Builder;
import ch.sahits.game.javafx.event.MouseClickLocationEventHandler;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.util.TextParser;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import ch.sahits.game.openpatrician.util.model.Text;
import ch.sahits.game.util.UIFactory;
import javafx.event.EventHandler;
import javafx.geometry.Insets;
import javafx.scene.Group;
import javafx.scene.effect.PerspectiveTransform;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.AnchorPane;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Color;
import javafx.scene.shape.Polygon;
import javafx.scene.shape.Rectangle;
import javafx.scene.text.Font;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;

/**
 * Scene displaying the hanseatic history.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Aug 10, 2013
 *
 */
@Prototype
public class HanseaticHistoryScene extends BaseStartupScene {
	private final Logger logger = Logger.getLogger(HanseaticHistoryScene.class);
	private static final int FONT_SIZE = 36;
	@Autowired
	private Locale locale;

	@Autowired
	private MessageSource messageSource;
	@Autowired
	@Qualifier("resourceReference")
	private MessageSource resources;


	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private TextParser textParser;
	@Autowired
	private IPolygonLoader polygonLoader;
	@Autowired
	private IFontLoader fontLoader;
	
	private Group labelShapes;
	
	private Group paginationGroup;

	public HanseaticHistoryScene(double width, double height) {
		super(new Pane(), width, height);
	}
	@Override
	protected String getStyleSheetFilename() {
		return "hanseaticHistory.css";
	}
	@PostConstruct
	private final void setUpScene() {
		Pane root = createBaseSceneLayout();
		Text text = getHistoryText();
		paginationGroup = new Group();
		paginationGroup.setManaged(false);
		Insets insets = new Insets(0, 10, 0, 70);
		Polygon backLabel = createPolygonShape("backButton");
		Polygon nextLabel = createPolygonShape("nextButton");
		Font defaultFont = fontLoader.createDefaultFont(FONT_SIZE);
		Font headerFont = fontLoader.createDefaultFont(FONT_SIZE + 6); // bold font not available
		PaginationV2 pagination = PaginationV2Builder.create()
				.backButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.back", new Object[]{}, locale.getCurrentLocal()))
				.nextButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.next", new Object[]{}, locale.getCurrentLocal()))
				.text(text)
				.contentMaxWidth(untransformedWidth)
				.contentMaxHeight(untransformedHeight)
				.padding(insets)
				.navigationLabelFont(defaultFont)
				.headerFont(headerFont)
				.paragraphFont(defaultFont)
				.centeredFont(defaultFont)
				.backButtonNode(backLabel)
				.nextButtonNode(nextLabel)
				.firstLastPageAction(new BackToStartupScene())
				.build();

		AnchorPane ap = new AnchorPane();
		ap.getChildren().add(paperScroll);
		AnchorPane.setBottomAnchor(paperScroll, 0.0);
		AnchorPane.setRightAnchor(paperScroll, 0.0);
		final Rectangle ph = new Rectangle(getWidth(), getHeight(), Color.TRANSPARENT);
		ph.widthProperty().bind(widthProperty());
		ph.heightProperty().bind(heightProperty());
		ap.getChildren().addAll(ph);

		paginationGroup.getChildren().add(pagination);
		
		pagination.setOnMousePressed(new MouseClickLocationEventHandler());
		
		labelShapes = new Group();
		labelShapes.setManaged(false);
		
		labelShapes.getChildren().addAll(backLabel, nextLabel);
		
		PerspectiveTransform transform = createPerspectiveTransform();
		
		pagination.setEffect(transform);

		root.getChildren().addAll(ap, paginationGroup, labelShapes);
	}
	/**
	 * Create the polygon using the polygon loader.
	 * @param polygonName name of the polygon
	 * @return Polygon definition
	 */
	private Polygon createPolygonShape(String polygonName) {
		List<ch.sahits.game.graphic.data.Polygon> startupPolygons = polygonLoader.getPolygon("/polygonDefinition.xml", "history");
        
		Polygon untransformed = polygonLoader.toJavaFXPolygon(polygonLoader.find(startupPolygons, polygonName));
		untransformed.setFill(Color.TRANSPARENT);
		return untransformed;
	}
	/**
	 * Retrieve the history text.
	 * @return
	 */
	private Text getHistoryText() {
		String resourceName = resources.getMessage("history.hanseatic_league", new Object[0], locale.getCurrentLocal());
		URL url = getClass().getClassLoader().getResource(resourceName);
		try {
			InputStream is= url.openStream();
			return textParser.parse(is);
		} catch (IOException e) {
			logger.warn("History text could not be found: " + url.toExternalForm());
			return new Text();
		}
	}
	
	@Override
	public void heightChange(double oldHeight, double newHeigth) {
		double top = Math.max(newHeigth - untransformedHeight - paginationGroup.getLayoutY() - 7, 0);
		paginationGroup.translateYProperty().set(top);
		if (oldHeight == 0 && newHeigth == getHeight()) {
			top = 0;
		} else {
			top = labelShapes.translateYProperty().get()+(newHeigth-oldHeight);
		}
		labelShapes.translateYProperty().set(top);
	}
	

	@Override
	public void widthChange(double oldWidth, double newWidth) {
		double left = Math.max(newWidth - untransformedWidth - paginationGroup.getLayoutX()-36, 0);
		paginationGroup.translateXProperty().set(left);
		if (oldWidth == 0 && newWidth == getWidth()) {
			left = 0;
		} else {
			left = labelShapes.translateXProperty().get()+(newWidth-oldWidth);
		}
		labelShapes.translateXProperty().set(left);
	}
	/**
	 * EvenHandler to get back to the startup screen.
	 * @author Andi Hotz, (c) Sahits GmbH, 2013
	 * Created on Sep 4, 2013
	 *
	 */
	private class BackToStartupScene implements EventHandler<MouseEvent> {

		@Override
		public void handle(MouseEvent arg0) {
			changeScene(uiFactory.getStartupScene(getWidth(), getHeight()));
			
		}
		
	}
}
