package ch.sahits.game.event.handler.impl;

import ch.sahits.game.event.ShipEntersPortEvent;
import ch.sahits.game.event.data.CourierPatrolStateInternal;
import ch.sahits.game.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.graphic.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Optional;
import com.google.common.eventbus.AsyncEventBus;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 16, 2014
 */
@Prototype
public class NextCourierPatrolStation  extends ShipEntersPortEventListener {
    /** Next station may be null if it is the last */
    private final CourierPatrolStateInternal state;
    private final IPlayer player;
    private final DateTime deadline;
    @Autowired
    private Date date;
    @Autowired
    private ClientTaskFactory taskFactory;
    @Autowired
    private Random rnd;

    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;

    /**
     * Create the next patrol station listener. This listener adds itself to the event queue and removes the first city from the
     * patrol state
     * @param ship
     * @param city
     * @param state
     * @param player
     * @param deadline
     */
    public NextCourierPatrolStation(IShip ship, ICity city, CourierPatrolStateInternal state, IPlayer player, DateTime deadline) {
        super(ship, city);
        this.state = state;
        this.player = player;
        this.deadline = deadline;
        state.removeFirstCity();
    }
    @PostConstruct
    private void initializeEventBus() {
        clientServerEventBus.register(this);
    }


    @Override
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
        DateTime now = date.getCurrentDate();
        if (now.isAfter(deadline)) {
            clientEventBus.post(new DisplayMessage(messageSource.getMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.unreliable", new Object[]{}, locale.getCurrentLocal())));
            getShip().leavePassanger();
            clientServerEventBus.unregister(this);
        }
    }

    @Override
    public boolean execute() {
        Optional<ICity> nextCity = state.nextDestination();
        if (!nextCity.isPresent()){ // last leg
            int premium = state.getPremium();
            player.updateCash(premium);
            getShip().leavePassanger();
            clientEventBus.post(new DisplayMessage(messageSource.getMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.salary", new Object[]{premium}, locale.getCurrentLocal())));
        } else { // next leg
            ICity next = nextCity.get();
            state.removeFirstCity();
            int minutes = rnd.nextInt(600);

            DateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
            String message = messageSource.getMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation", new Object[]{next.getName()}, locale.getCurrentLocal());
            taskFactory.getPostponedDisplayMessage(postMessage, message);
            DateTime deadline = date.getCurrentDate().plusDays(6);
            new NextCourierPatrolStation(getShip(), state.nextDestination().get(), state, player, deadline);
        }
        return true;
    }
}
