package ch.sahits.game.event.handler.impl;

import ch.sahits.game.event.ShipEntersPortEvent;
import ch.sahits.game.event.data.PatrolInternalState;
import ch.sahits.game.event.handler.ShipEntersPortEventListener;
import ch.sahits.game.graphic.event.task.ClientTaskFactory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.base.Optional;
import com.google.common.eventbus.AsyncEventBus;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.Random;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 21, 2014
 */
@Prototype
class NextPatrolStation extends ShipEntersPortEventListener {
    /** Next station may be null if it is the last */
    private final PatrolInternalState state;
    private final IPlayer player;
    private final DateTime deadline;
    private final AttackListener atackListener;
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private Date date;
    @Autowired
    private Random rnd;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private ClientTaskFactory taskFactory;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;

    /**
     * Create the next patrol station listener. This listener adds itself to the event queue and removes the first city from the
     * patrol state.
     * @param ship on patrol
     * @param city next city
     * @param state patrol state
     * @param player current player (ship owner)
     * @param deadline until the ship has to arrive
     * @param atackListener attack listener
     */
    public NextPatrolStation(IShip ship, ICity city, PatrolInternalState state, IPlayer player, DateTime deadline, AttackListener atackListener) {
        super(ship, city);
        this.state = state;
        this.player = player;
        this.deadline = deadline;
        this.atackListener = atackListener;
        state.removeFirstCity();

    }
    @PostConstruct
    private void initEventBus() {
        clientServerEventBus.register(this);
    }

    @Override
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        super.handleShipEntersPort(event);
        DateTime now = date.getCurrentDate();
        if (now.isAfter(deadline)) {
            clientEventBus.post(new DisplayMessage(messageSource.getMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.unreliable", new Object[]{}, locale.getCurrentLocal())));
            clientServerEventBus.unregister(this);
            clientServerEventBus.unregister(atackListener);
        }
    }


    @Override
    public boolean execute() {
        Optional<ICity> nextCity = state.nextDestination();
        if (!nextCity.isPresent()){ // last leg
            int premium = state.getPremium()+state.getNumberOfAttacks()*state.getBonusPerAttack();
            player.updateCash(premium);
            clientEventBus.post(new DisplayMessage(messageSource.getMessage("ch.sahits.game.event.handler.impl.NextPatrolStation.premium", new Object[]{premium}, locale.getCurrentLocal())));
            clientServerEventBus.unregister(atackListener);
        } else { // next leg
            ICity next = nextCity.get();
            state.removeFirstCity();
            int minutes = rnd.nextInt(600);

            DateTime postMessage = date.getCurrentDate().plusMinutes(minutes);
            String message = messageSource.getMessage("ch.sahits.game.event.handler.impl.NextCourierPatrolStation.nextStation", new Object[]{next.getName()}, locale.getCurrentLocal());
            taskFactory.getPostponedDisplayMessage(postMessage, message);
            DateTime deadline = date.getCurrentDate().plusDays(6);
            new NextPatrolStation(getShip(), state.nextDestination().get(), state, player, deadline, atackListener);
        }
        return true;
    }}
