package ch.sahits.game.graphic.display;

import ch.sahits.game.event.EGameStatusChange;
import ch.sahits.game.event.GameStateChange;
import ch.sahits.game.graphic.display.scene.BaseStartupScene;
import ch.sahits.game.javafx.OpenPatricianScene;
import ch.sahits.game.javafx.event.AggregateEventHandler;
import ch.sahits.game.javafx.event.CloseApplicationEventHandler;
import ch.sahits.game.javafx.event.MouseClickLocationEventHandler;
import ch.sahits.game.openpatrician.server.ServerLauncher;
import ch.sahits.game.openpatrician.util.GamePropertyUtility;
import ch.sahits.game.util.UIFactory;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.application.Application;
import javafx.event.EventHandler;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyEvent;
import javafx.stage.Stage;
import org.apache.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;

/**
 * This is the starting class of the client side UI thread (FX Thread).
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jul 28, 2013
 *
 */
public final class OpenPaticianApplicationWindow  extends Application implements SceneChangeable {
	private final Logger logger = Logger.getLogger(getClass());

	private GamePropertyUtility gamePropertyUtility;
	private final int minWidth;
	private final int minHeight;
	private int width;
	private int height;
	private boolean fullscreen;
	private Stage primaryStage;

	private final AggregateEventHandler<KeyEvent> keyEventHandlerAggregate;
	private final MouseClickLocationEventHandler mouseClickEventHandler;

    private static OpenPaticianApplicationWindow instance;
	private ApplicationContext context;

	/**
     *  Constructor required by the supper class to create an instance of the application.
     */
	public OpenPaticianApplicationWindow() {
		Preconditions.checkState(instance == null, "Application should only be initialized once.");
        minHeight = 766;
        minWidth = 1074;
		width = minWidth;
		height = minHeight;
		this.fullscreen = false;
		keyEventHandlerAggregate = new AggregateEventHandler<>();

		CloseApplicationEventHandler closeEventHandler = new CloseApplicationEventHandler();
		mouseClickEventHandler = new MouseClickLocationEventHandler();
		EventHandler<KeyEvent> fullScreenEventHandler = new EventHandler<KeyEvent>(){

			@Override
			public void handle(KeyEvent event) {
				if (event.getCode().equals(KeyCode.F) && event.isControlDown()) {
					updateFullscreenMode();
				}
			}
		};
		keyEventHandlerAggregate.addEventHandler(closeEventHandler);
		keyEventHandlerAggregate.addEventHandler(fullScreenEventHandler);

        instance = this;
	}

    public static SceneChangeable getApplication() {
        return instance;
    }
	
	@Override
	public void start(Stage primaryStage) throws Exception {
		this.primaryStage = primaryStage;
		this.primaryStage.setMinWidth(minWidth);
		this.primaryStage.setMinHeight(minHeight);
		context = ServerLauncher.getClientServerContext();
		UIFactory uiFactory = context.getBean(UIFactory.class);
		BaseStartupScene startupS = uiFactory.getStartupScene(width, height);
		gamePropertyUtility = context.getBean(GamePropertyUtility.class);
		this.fullscreen = Boolean.valueOf((String) gamePropertyUtility.getProperties().get("window.fullscreen"));
		startupS.setSceneChangeable(this);
		startupS.setOnMousePressed(mouseClickEventHandler);
		startupS.setOnKeyPressed(keyEventHandlerAggregate);
		primaryStage.setFullScreen(fullscreen);
        primaryStage.setTitle("OpenPatrician");
        primaryStage.setScene(startupS);
        primaryStage.show();
	}
	/**
	 * @see ch.sahits.game.graphic.display.SceneChangeable#changeScene(ch.sahits.game.javafx.OpenPatricianScene)
	 */
	@Override
	public void changeScene(final OpenPatricianScene scene) {
		scene.setOnMousePressed(mouseClickEventHandler);
		scene.setOnKeyPressed(keyEventHandlerAggregate);
		primaryStage.setScene(scene);
	}
	/**
	 * Toggle between full screen and non full screen mode.
	 */
	public void updateFullscreenMode() {
		fullscreen = !fullscreen;
		primaryStage.setFullScreen(fullscreen);
	}

    /**
     * Starting point of the client side application thread.
     * This method must be static and called in that way, for the
     * application to work.
     * @param args Array of arguments, currently empty.
     */
	public static void startClientUI(String[] args) {
		launch(args);
	}

	@Override
	public double getSceneWidth() {
		return primaryStage.getScene().getWidth();
	}

	@Override
	public double getSceneHeight() {
		return primaryStage.getScene().getHeight();
	}

    @Override
    public void stop() throws Exception {
        System.out.println("Stopping the UI Application");

        stopUIApplicationContext();
        super.stop();
    }

	/**
	 * Closing the application context for the user interface.
	 */
	private void stopUIApplicationContext() {
		AsyncEventBus eventBus = (AsyncEventBus) context.getBean("serverClientEventBus");
		eventBus.post(new GameStateChange(EGameStatusChange.SHUTDOWN));
		((AbstractApplicationContext)context).close();
	}
}
