package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.service.DecoratedTextFactory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.engine.land.city.LoanerEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICreditor;
import ch.sahits.game.openpatrician.model.city.impl.Debt;
import ch.sahits.game.openpatrician.model.city.impl.ILoanProposal;
import ch.sahits.game.openpatrician.model.city.impl.LoanerState;
import ch.sahits.game.openpatrician.model.impl.Citizen;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Random;

/**
 * Dialog for taking up a loan.
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Jan 04, 2015
 */
@Prototype
public class LoanerTakeLoanDialog extends CloseButtonDialog {
    private final int BUTTON_X = (WIDTH - 124) / 2;
    private final int BUTTON_Y = CLOSE_BTN_Y_POS - 24;
    public static final String DEFAULT_NON_SELECTED_STYLE = "-fx-fill: dimgray; -fx-text-fill: dimgray;";
    public static final String SELECTED_STYLE = "-fx-font-weight: bolder; -fx-fill: black;  -fx-text-fill: black;";

    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    private LoanerEngine loanerEngine;
    @Autowired
    private DecoratedTextFactory textFactory;
    @Autowired
    private Random rnd;
    @Autowired
    private Date date;
    @Autowired
    private ClientViewState viewState;

    private LoanerState loaner;
    private final ICityPlayerProxyJFX city;
    private final Font font;

    private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
    private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");

    private ObjectProperty<ILoanProposal> selectedProposal = new SimpleObjectProperty<>(null);
    private VBox box;

    public LoanerTakeLoanDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
    }
    @PostConstruct
    private void initializeDialog() {
        loaner = (LoanerState) loanerEngine.getLoaner(city.getCity());
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.notice.NoticeBoardFactory.takeLoan", new Object[]{}, locale.getCurrentLocal()));
        String header = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.dialogSubheading", new Object[]{}, locale.getCurrentLocal());
        box = new VBox(5);
        box.setLayoutX(2 * FRAME_BORDER);
        box.setLayoutY(200);
        Text subHeading = new Text();
        subHeading.setText(header);
        subHeading.setFont(font);
        box.getChildren().add(subHeading);
        for (final ILoanProposal proposal : loaner.getLoanProposals()) {
            int amount = proposal.getAmount();
            int payback = (int) (amount * proposal.getInterest());
            int duration = proposal.getDurationInWeeks();
            String interest = String.format("%.1f", (proposal.getInterest() - 1) * 100);
            String s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.loanProposal", new Object[]{amount, duration, payback, interest}, locale.getCurrentLocal());
            final DecoratedText entry = textFactory.createDecoratedText(s, new HashMap<String, Object>());
            entry.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    selectedProposal.setValue(proposal);
                    deactivateAll();
                    entry.applyStyle(SELECTED_STYLE);
                }
            });
            entry.applyStyle(DEFAULT_NON_SELECTED_STYLE);
            box.getChildren().add(entry);
        }

        final OpenPatricianLargeWaxButton acceptBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.LoanerTakeLoanDialog.btnLabel", new Object[]{}, locale.getCurrentLocal()), font);
        acceptBtn.setLayoutX(BUTTON_X);
        acceptBtn.setLayoutY(BUTTON_Y);
        acceptBtn.setDisable(true);
        selectedProposal.addListener(new ChangeListener<ILoanProposal>() {
            @Override
            public void changed(ObservableValue<? extends ILoanProposal> observableValue, ILoanProposal oldProposal, ILoanProposal newProposal) {
                if (newProposal != null) {
                    acceptBtn.setDisable(false);
                }
            }
        });
        acceptBtn.setOnAction(createAcceptHandler());
        getContent().addAll(box, acceptBtn);
    }
    private void deactivateAll() {
        for (Node node : box.getChildren()) {
            if (node instanceof DecoratedText) {
                ((DecoratedText)node).applyStyle(DEFAULT_NON_SELECTED_STYLE);
            }
        }
    }
    private EventHandler<MouseEvent> createAcceptHandler() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                DateTime now = date.getCurrentDate();
                ILoanProposal proposal = selectedProposal.get();
                IPlayer debitor = city.getPlayer();
                ESocialRank rank = ESocialRank.values()[rnd.nextInt(ESocialRank.values().length-2)]; // Exclude Mayor and Alderman
                ICreditor creditor = new Citizen(lastNameLoader.getRandomName(), firstNameLoader.getRandomName(), rank, city.getCity());
                DateTime dueDate = now.plusWeeks(proposal.getDurationInWeeks());
                Debt debt = Debt.builder()
                        .creditor(creditor)
                        .amount(proposal.getAmount())
                        .dueDate(dueDate)
                        .interest(proposal.getInterest())
                        .debitor(debitor)
                        .build();
                loaner.addDebt(debt);
                loaner.getLoanProposals().remove(proposal);
                debitor.getCompany().updateCash(proposal.getAmount());
                executeOnCloseButtonClicked();
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_LOANER);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
