package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.IPersonalData;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.scene.Node;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;

/**
 * Dialog presenting personal data.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 7, 2013
 *
 */
@Prototype
public class PersonalDialog extends CloseButtonDialog {
	private Font font;
	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
	@Autowired
	private Date date;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
	@Autowired
	private Locale locale;

	@Autowired
	private MessageSource messageSource;
	@Autowired
	private ModelTranslations modelTranslator;

	public PersonalDialog(Font font, ICityPlayerProxyJFX city) {
		super(font);
		this.font = font;
		this.city = city;
	}
	@PostConstruct
	private void initializeDialog() {
		setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.personal", new Object[]{}, locale.getCurrentLocal()));
		IPlayer player = city.getPlayer();
		IPersonalData personalData = player.getPersonalData();
		
		VBox box = new VBox();
		box.setLayoutX(2*FRAME_BORDER);
		box.setLayoutY(80);

		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.name", new Object[]{}, locale.getCurrentLocal()));
		
		StringBuilder sb = new StringBuilder();
		sb.append(personalData.getName()).append(" ").append(personalData.getLastName());
		ESocialRank rank = player.getRank();
		sb.append(" (").append(modelTranslator.getLocalDisplayName(player.getCareerLevel())).append(" ").append(modelTranslator.getLocalDisplayName(rank)).append(")");
		createAndAddNewText(box, sb.toString());
		
		box.getChildren().add(createEmptyNode());
		
		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.born", new Object[]{}, locale.getCurrentLocal()));
		
		sb = new StringBuilder();
		ICity birthPlace = personalData.getBirthPlace();
		DateTime birthDate = personalData.getBirthDate();
		sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.on", new Object[]{}, locale.getCurrentLocal())).append(date.todisplayString(birthDate.getDayOfMonth(), birthDate.getMonthOfYear(), birthDate.getYear())).append(" in ").append(birthPlace.getName()).append(", ");
		sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.age", new Object[]{}, locale.getCurrentLocal())).append(personalData.getAge(date.getCurrentDate()));
		createAndAddNewText(box, sb.toString());

		box.getChildren().add(createEmptyNode());
		
		createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.hometown", new Object[]{}, locale.getCurrentLocal()));
		
		createAndAddNewText(box, player.getHometown().getName());

		box.getChildren().add(createEmptyNode());

		sb = new StringBuilder();
		sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.socStatus", new Object[]{}, locale.getCurrentLocal())).append(city.getCity().getSocialRank());
		createAndAddNewText(box, sb.toString());
		
		createAndAddNewText(box, getReputationText(player));
		
		List<String> careerAdvice = getCareerAdvice(player);
		if (careerAdvice.isEmpty()) {
			box.getChildren().add(createEmptyNode());
			box.getChildren().add(createEmptyNode());
		} else {
			createAndAddNewText(box, careerAdvice.get(0));
			createAndAddNewText(box, careerAdvice.get(0));
		}
		
		createAndAddNewText(box, getUnderworldReputation(player));
		
		ISpouseData spouseData = player.getSpouseData();
		if (spouseData != null) {
			String s;
			if (spouseData.isMale()){
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.husbandName", new Object[]{}, locale.getCurrentLocal());
			} else {
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.wifeName", new Object[]{}, locale.getCurrentLocal());
			}
			createAndAddNewText(box, s);
			
			sb = new StringBuilder();
			sb.append(spouseData.getName()).append(" ").append(spouseData.getLastName());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.age2", new Object[]{}, locale.getCurrentLocal())).append(spouseData.getAge(date.getCurrentDate()));
			createAndAddNewText(box, sb.toString());

			box.getChildren().add(createEmptyNode());

			createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.born", new Object[]{}, locale.getCurrentLocal()));

			sb = new StringBuilder();
			birthDate = spouseData.getBirthDate();
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.on", new Object[]{}, locale.getCurrentLocal())).append(date.todisplayString(birthDate.getDayOfMonth(), birthDate.getMonthOfYear(), birthDate.getYear()))
			.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.in", new Object[]{}, locale.getCurrentLocal())).append(spouseData.getBirthPlace().getName());
			createAndAddNewText(box, sb.toString());

			createAndAddNewText(box, messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.children", new Object[]{}, locale.getCurrentLocal()));
			createAndAddNewText(box, String.valueOf(spouseData.getNumberOfChildren()));
			List<String> spouseReputation = getSpouseReputation(player, spouseData);
			createAndAddNewText(box, spouseReputation.get(0));
			createAndAddNewText(box, spouseReputation.get(1));
		} // end spouse data


		getContent().add(box);
	}
	private void createAndAddNewText(VBox box, String s) {
		Text spouseName = new Text(s);
		spouseName.setFont(font);
		box.getChildren().add(spouseName);
	}
	
	private Node createEmptyNode() {
		Text text = new Text("");
		text.setFont(font);
		return text;
	}
	
	private String getReputationText(IPlayer player) {
		IReputation rep = city.getCity().getReputation(player);
        // fixme: andi 02/05/14: reputation is null
		int repDiff = rep.getReputationDifference();
		StringBuilder sb = new StringBuilder();
		if (repDiff<0){
			// reputation has sunk
			if (repDiff<=-10 && repDiff>-25){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff>-10){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else if (repDiff>0){
			// reputation has risen
			if (repDiff>=10 && repDiff<25){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff<10){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else {
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.unchangedRep", new Object[]{}, locale.getCurrentLocal()));
		}
		return sb.toString();
	}
	/**
	 * Get the two lines of text for career advice.
	 * @param player
	 * @return
	 */
	private List<String> getCareerAdvice(IPlayer player){
		ESocialRank rank = player.getRank();
		ArrayList<String> lines = new ArrayList<>();
		if (rank!=ESocialRank.ALDERMAN && rank!=ESocialRank.MAYOR){
			// draw string what is needed
			ESocialRank nextRank = rank.getNextRank();
			boolean needFortune = false;
			if (player.getCompany().getCompanyValue()<nextRank.getCompanyValue()){
				needFortune=true;
			}
			boolean needReputation=false;
			if (city.getCity().getReputation(player).getPopularity()<nextRank.getSocialReputation()){
				needReputation=true;
			}
			StringBuilder sb = new StringBuilder();
			if (needFortune && needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needFortune){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice2_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice3_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			}
		}
		return lines;
	}
	
	public List<String> getSpouseReputation(IPlayer player, ISpouseData spouse) {
		ArrayList<String> lines = new ArrayList<>();
		StringBuilder sb = new StringBuilder();
		if (city.getCity()==spouse.getBirthPlace()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else if (city.getCity()==player.getCompany().getHomeTown()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else {
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		}
		return lines;
	}
	
	private String getUnderworldReputation(IPlayer player) {
		ICareer careerLevel = player.getCareerLevel();
		String s;
		if (careerLevel instanceof IMilitantCareer){
			if (((EMilitantCareer)careerLevel)!=EMilitantCareer.BOISTEROUS &&
					((EMilitantCareer)careerLevel)!=EMilitantCareer.BOLD ){
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep1", new Object[]{}, locale.getCurrentLocal());
			} else {
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
			}
		} else {
			s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
		}
		return s;
	}
	@Override
	protected void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}

}
