package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.event.ViewChangeEvent;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.util.TavernDialogUtil;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.graphic.display.notice.ENoticeItem;
import ch.sahits.game.graphic.javafx.display.MainGameView;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.IConcurrent;
import ch.sahits.game.openpatrician.model.people.ICourier;
import ch.sahits.game.openpatrician.model.people.IEscorte;
import ch.sahits.game.openpatrician.model.people.IFugitive;
import ch.sahits.game.openpatrician.model.people.IPatrol;
import ch.sahits.game.openpatrician.model.people.IPirateHunter;
import ch.sahits.game.openpatrician.model.people.ISideRoomPerson;
import ch.sahits.game.openpatrician.model.people.ISmuggler;
import ch.sahits.game.openpatrician.model.people.IThieve;
import ch.sahits.game.openpatrician.model.people.ITransportTrader;
import ch.sahits.game.openpatrician.model.people.ITreasureMapOwner;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.application.Platform;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.DoubleBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 22, 2013
 */
public abstract class TavernBaseSideRoomPersonDialog extends CloseButtonDialog {
    protected final ICityPlayerProxyJFX city;
    protected final Font font;
    private final TavernState tavernState;
    protected final ObjectProperty<ISideRoomPerson> currentPerson;
    @Autowired
    protected Date date;
    @Autowired
    protected TavernDialogUtil tavernDialogUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    protected Locale locale;

    @Autowired
    protected MessageSource messageSource;

    private final int BUTTON_X = (WIDTH - 124) / 2;
    private final int LOWER_BUTTON_Y = CLOSE_BTN_Y_POS - 24;
    private final int UPPER_BUTTON_Y = LOWER_BUTTON_Y - 24;
    private OpenPatricianLargeWaxButton acceptBtn;


    public TavernBaseSideRoomPersonDialog(Font font, ICityPlayerProxyJFX city, ISideRoomPerson person) {
        super(font);
        this.city = city;
        this.font = font;
        tavernState = city.getCity().getCityState().getTavernState();
        currentPerson =  new SimpleObjectProperty<>(this, "currentPerson", person);
        // todo: andi 23/12/13: person may leave
    }
    protected final void initializeButtons(){
        final BooleanBinding hasMorePersons = hasNextPersonBinding();
        acceptBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()), font);
        acceptBtn.setLayoutX(BUTTON_X);
        acceptBtn.setOnAction(createAcceptHandler());
        OpenPatricianLargeWaxButton nextBtn = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.next", new Object[]{}, locale.getCurrentLocal()), font);
        nextBtn.setLayoutX(BUTTON_X);
        nextBtn.setLayoutY(LOWER_BUTTON_Y);
        nextBtn.setOnAction(createNextPersonHander());
        nextBtn.visibleProperty().bind(hasMorePersons);
        acceptBtn.layoutYProperty().bind(new DoubleBinding() {
            {
                super.bind(hasMorePersons);
            }
            @Override
            protected double computeValue() {
                if (hasMorePersons.get()) {
                    return UPPER_BUTTON_Y;
                } else {
                    return LOWER_BUTTON_Y;
                }
            }
        });
        final BooleanBinding disableAcceptButton = disableAcceptButton();
        acceptBtn.setDisable(disableAcceptButton.get());
        disableAcceptButton.addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observableValue,
                                Boolean oldValue, Boolean newValue) {
                acceptBtn.setDisable(newValue);
            }
        });
        getContent().addAll(nextBtn, acceptBtn);
    }
    protected void removeAcceptButton() {
        Platform.runLater(new Runnable() {
            @Override
            public void run() {
                getContent().remove(acceptBtn);
            }
        });
    }

    /**
     * Define the action that is executed on hitting the accept button
     * @return
     */
    protected abstract EventHandler<MouseEvent> createAcceptHandler();

    /**
     * Create the event handler to display the next person
      * @return
     */
    private EventHandler<MouseEvent> createNextPersonHander() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                tavernState.talkingToProperty().get().remove(currentPerson.get());
                List<ISideRoomPerson> persons = tavernState.presentPersonsInSideRoomBinding().get();
                for (int i = 0; i < persons.size(); i++) {
                    ISideRoomPerson person = persons.get(i);
                    if (person.equals(currentPerson.get())) {
                        if (i == persons.size() - 1) {
                            currentPerson.set(persons.get(0));
                        } else {
                            currentPerson.set(persons.get(i + 1));
                        }
                        break;
                    }
                } // end for
                tavernState.talkingToProperty().get().put(currentPerson.get(), city.getPlayer());
                ENoticeItem nextPerson = map(currentPerson.get());
                clientEventBus.post(new ViewChangeEvent(MainGameView.class, nextPerson));
            }
        };
    }

    /**
     * Binding indicating if there is another person in the side room.
     * @return
     */
    private BooleanBinding hasNextPersonBinding() {
        return new BooleanBinding() {
            {
                super.bind(tavernState.presentPersonsInSideRoomBinding());
            }
            @Override
            protected boolean computeValue() {
                return tavernState.presentPersonsInSideRoomBinding().get().size() > 1;
            }
        };
    }

    private ENoticeItem map(ISideRoomPerson person) {
        if (person instanceof ITransportTrader) {
            return ENoticeItem.TAVERN_TRANSPORT_TRADER;
        }
        if (person instanceof IBuyer) {
           return ENoticeItem.TAVERN_BUYER;
        }
        if (person instanceof IEscorte) {
            return ENoticeItem.TAVERN_ESCORTE;
        }
        if (person instanceof IFugitive) {
           return ENoticeItem.TAVERN_FUGATIVE;
        }
        if (person instanceof ICourier) {
            return ENoticeItem.TAVERN_COURIER;
        }
        if (person instanceof IPatrol) {
            return ENoticeItem.TAVERN_PATROL;
        }
        if (person instanceof IThieve) {
            return ENoticeItem.TAVERN_BURGLAR;
        }
        if (person instanceof IConcurrent) {
            return ENoticeItem.TAVERN_CONCURRENT;
        }
        if (person instanceof IPirateHunter) {
            return ENoticeItem.TAVERN_PIRATE_HUNTER;
        }
        if (person instanceof IWarehouseTenant) {
            return ENoticeItem.TAVERN_WAREHOUSE_TENANT;
        }
        if (person instanceof ISmuggler) {
           return ENoticeItem.TAVERN_SMUGGLER;
        }
        if (person instanceof ITreasureMapOwner) {
            return ENoticeItem.TAVERN_TREASUREMAP_OWNER;
        }
        throw new RuntimeException("Unhandled side room person: "+person.getClass().getName());
    }


    /**
     * Binding defining the disabling of the accept button.
     * Binding evaluates to true when the accept button should be disabled.
     * @return
     */
    protected abstract BooleanBinding disableAcceptButton();

    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        tavernDialogUtil.stopTalkingToPerson(tavernState, city.getPlayer());
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

}
