package ch.sahits.game.graphic.display.dialog.action;

import ch.sahits.game.graphic.display.dialog.util.ITransferableJFX;
import ch.sahits.game.javafx.bindings.ConstantIntegerBinding;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.google.common.base.Preconditions;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.util.Pair;

/**
 * Action of buying stuff from the city onto the ship
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 23, 2011
 *
 */
class City2ShipJFXAction implements Runnable{
	private final IWare ware;
	private final ITransferableJFX dialog;
//	private static final Logger logger = Logger.getLogger(City2ShipAction.class);

	public City2ShipJFXAction(IWare ware, ITransferableJFX dialog) {
		super();
		this.ware = ware;
		this.dialog=dialog;
	}

	@Override
	public void run() {
		ICityPlayerProxyJFX city = dialog.getCityPlayerProxy();
		int availableAmountCity = city.getCity().getWare(ware).getAmount();
		if (availableAmountCity>0) {
			int amount2Move = dialog.getAmount(availableAmountCity); // This is ware specific size
			// consider available capacity
			final short sizeAsBarrels = ware.getSizeAsBarrels();
			ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
			if (sizeAsBarrels == 1) {
				if (office == null) {   // todo: andi 12/13/14: adjust for optinal return value
					amount2Move = Math.min(amount2Move, city.getActiveShip().getCapacity());
				}
			} else {
				int temp = Math.min(amount2Move * sizeAsBarrels, city.getActiveShip().getCapacity());
				amount2Move = temp / sizeAsBarrels;
			}
			int avgPrice = ware.buyPrice(new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(amount2Move));
			// check if this is afforable
			final long cash = city.getPlayer().getCompany().getCash();
			Pair<Integer, Integer> affordable = calculateAffordableAmount(cash, amount2Move, avgPrice, availableAmountCity);

			amount2Move = affordable.getKey();
			avgPrice = affordable.getValue();
			if (amount2Move <= 0) {
				return; // cannot buy anything
			}
			int movedAmount = city.getCity().move(ware, -amount2Move, city.getPlayer());
			if (amount2Move != -movedAmount) { // not enough available in city
				avgPrice = ware.buyPrice(new SimpleIntegerProperty(city.getCity().getWare(ware).getAmount() + movedAmount),
						new ConstantIntegerBinding(movedAmount));
				amount2Move = -movedAmount;
			}
			int loaded;
			if (office == null){  // amount2Move is max the ships capacity
				loaded = city.getActiveShip().load(ware, amount2Move, avgPrice);
			} else {
				final int capacity = city.getActiveShip().getCapacity();
				if (amount2Move > capacity) {
				   int loadOntoShip = capacity;
					int storeInOffice = amount2Move - capacity;
					loaded = city.getActiveShip().load(ware, loadOntoShip, avgPrice);
					loaded += office.move(ware, storeInOffice);
				} else {
					loaded = city.getActiveShip().load(ware, amount2Move, avgPrice);
				}
			}
			city.getPlayer().updateCash(-avgPrice*loaded);
		} // end no ware available
	}

	/**
	 * Calculate how much can be bought due to monetary restrictions.
	 * @param cash available cash
	 * @param desiredAmount amount that is desired to be bought
	 * @param avgPrice average price for an item when buying the whole amount.
	 * @param availableAmountCity amount of the ware available in the city
	 * @return Pair containing the affordable amount and the average price.
	 */
	Pair<Integer, Integer> calculateAffordableAmount(long cash, int desiredAmount, int avgPrice, int availableAmountCity) {
		if (cash < (long) avgPrice * desiredAmount) {
			int amountAprox = (int) (cash / avgPrice); // approx amount we can afford
			// fixme: andi 12/13/14: this might still be above the available cash
			if (amountAprox > 0) {
				int tempPrice = ware.buyPrice(new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(amountAprox));
				Preconditions.checkArgument(amountAprox*tempPrice<=cash, "Buying fewer items never results in a higher avg price");
				while (amountAprox * tempPrice + tempPrice < cash) {
					int newTempPrice = ware.buyPrice(new SimpleIntegerProperty(availableAmountCity), new ConstantIntegerBinding(++amountAprox));
					if (amountAprox * newTempPrice > cash) {
						// we cannot afford another item
						break;
					}
					tempPrice = newTempPrice;
				}
				return new Pair<>(amountAprox, tempPrice);
			} else {
				return new Pair<>(0, 0);
			}
		} else {
			return new Pair<>(desiredAmount, avgPrice);
		}
	}

}
