package ch.sahits.game.graphic.display.scene;

import ch.sahits.game.graphic.display.SceneChangeable;
import ch.sahits.game.graphic.image.IImageLoader;
import ch.sahits.game.javafx.OpenPatricianScene;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.scene.Parent;
import javafx.scene.control.Label;
import javafx.scene.effect.PerspectiveTransform;
import javafx.scene.effect.PerspectiveTransformBuilder;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.layout.Pane;
import javafx.scene.paint.Paint;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * Base class for all startup scenes featuring the paper scroll.
 * To get the writing onto the scroll the effect {@link #createPerspectiveTransform()}
 * must be applied. This effect however does not change the bounds
 * of the involved controls. Therefore the following steps have to be taken:
 * <ol>
 * <li>Define the bounds of the transformed control in polygonDefinition.xml</li>
 * <li>Create a Polygon
 * <pre>
 *      // Load the polygons for the scene with the name startup from the indicated file
 * 		List<ch.sahits.game.graphic.data.Polygon> startupPolygons = polygonLoader.getPolygon("/polygonDefinition.xml", "startup");
 *      // Find the polygon with the name polygonName and create an instance
 *  	Polygon untransformed = polygonLoader.toJavaFXPolygon(polygonLoader.find(startupPolygons, polygonName));
 * </pre>
 * </li>
 * <li>Add a change listener to the hoverProperty of the Polygon. The change listener updates the appearance of the of the control</li>
 * <li>Add a mouse clicked listener to handle click events</li>
 * <li>Add all polygons into an unmanaged group and add them to the root element</li>
 * </ol>
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Aug 10, 2013
 *
 */
public abstract class BaseStartupScene extends OpenPatricianScene {



	@Autowired
    @Qualifier("startScreenImageLoader")
	private IImageLoader imsLoader;
	protected ImageView paperScroll;
	protected double imgWidth;
	protected double imgHeigth;
	protected double untransformedWidth;
	protected double untransformedHeight;
	private SceneChangeable sceneChangeable;

	public BaseStartupScene(Parent root, double width, double height,
			boolean depthBuffer) {
		super(root, width, height, depthBuffer);
	}

	public BaseStartupScene(Parent root, double width, double height, Paint fill) {
		super(root, width, height, fill);
	}

	public BaseStartupScene(Parent root, double width, double height) {
		super(root, width, height);
	}

	public BaseStartupScene(Parent root, Paint fill) {
		super(root, fill);
	}

	public BaseStartupScene(Parent root) {
		super(root);
	}
	/**
	 * Define the stylesheet name to be applied for this scene.
	 * @return
	 */
	protected abstract String getStyleSheetFilename();

	/**
	 * Create the base scene layout containing the paper scroll.
	 * Add the style sheet defined in {@link #getStyleSheetFilename()}. Initialize dimensions
	 * @return Root stack pane.
	 */
	protected final Pane createBaseSceneLayout() {
		// Paper scroll	
		Image img = imsLoader.getImage("PaperScroll");
		imgWidth = img.getWidth();
		imgHeigth = img.getHeight();
		paperScroll = new ImageView(img);
		Pane root = (Pane)getRoot();
		getStylesheets().add(this.getClass().getResource(getStyleSheetFilename()).toExternalForm());
		untransformedWidth = 560;
		untransformedHeight = 507;
		return root;
	}

	/**
	 * Define the perspective transform that will map a rectangle (control) to the paper scroll.
	 * @return transfrom effect to be applied onto the control
	 */
	protected final PerspectiveTransform createPerspectiveTransform() {
		PerspectiveTransform perspectiveTransform = PerspectiveTransformBuilder.create()
				.ulx(185).uly(0)
				.urx(449).ury(19)
				.llx(-20).lly(403)
				.lrx(459).lry(502)
				.build();
		return perspectiveTransform;
	}
	/**
	 * ChangeListener which changes the style class of the label upon a boolean change.
	 * @author Andi Hotz, (c) Sahits GmbH, 2013
	 * Created on Aug 10, 2013
	 *
	 */
	protected static class LabelHoverChangeListener implements ChangeListener<Boolean> {
		
		private final Label label;
		

		public LabelHoverChangeListener(Label label) {
			super();
			this.label = label;
		}


		@Override
		public void changed(ObservableValue<? extends Boolean> observable,
				Boolean oldValue, Boolean newValue) {
			if (newValue) {
				label.getStyleClass().remove("label-unfocused");
				label.getStyleClass().add("label-focus");
			} else {
				label.getStyleClass().remove("label-focus");
				label.getStyleClass().add("label-unfocused");
			}
			
		}
	}
	public SceneChangeable getSceneChangeable() {
		return sceneChangeable;
	}

	public void setSceneChangeable(SceneChangeable sceneChangeable) {
		this.sceneChangeable = sceneChangeable;
	}

	public void changeScene(OpenPatricianScene scene) {
		double oldWidth = 0;
		double oldHeight = 0;
		double newWidth = sceneChangeable.getSceneWidth();
		double newHeight = sceneChangeable.getSceneHeight();
		sceneChangeable.changeScene(scene);
		// Fix needed to avoid bug https://javafx-jira.kenai.com/browse/RT-30392
		scene.widthChange(oldWidth, newWidth); // old value is wrong as that is the same as the new one
		scene.heightChange(oldHeight, newHeight);
		if (scene instanceof BaseStartupScene) {
			((BaseStartupScene)scene).setSceneChangeable(sceneChangeable); // propagate
		}
	}
	
}