package ch.sahits.game.graphic.display;

import ch.sahits.game.event.data.NewGameClient;
import ch.sahits.game.graphic.display.model.CityPlayerProxyJFX;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.OptionalType;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import java.util.Optional;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Getter;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * This is the state that represents the actual state of what the client currently displayes.
 * This differeciates between the actual states:
 * <ul>
 *     <li>City</li>
 *     <li>Map</li>
 *     <li>Sea</li>
 * </ul>
 * As well for the city there is additional information on which city.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 15, 2014
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN,EClassCategory.SINGLETON_BEAN})
public class ClientViewState {
    @XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());

    /** Main state */
    @Getter
    private EViewState state;
    /** proxy for the city is only present when the state is city */
    @Getter
    @OptionalType(ICityPlayerProxyJFX.class)
    private Optional<ICityPlayerProxyJFX> currentCityProxy;
    /** The player behind the client */
    @Getter
    private IPlayer player;

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @PostConstruct
    private void initEventBus() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        logger.debug("Unregister event bus in PreDestroy");
        clientServerEventBus.unregister(this);
    }

    /**
     * Initialize the player that goes with this client. The player does not change later on,
     * @param newGameDTO
     */
    @Subscribe
    public void initializeState(NewGameClient newGameDTO) {
        this.player = newGameDTO.getPlayer();
        state = EViewState.CITY;
        IShip activeShip = null;
        for (IShip ship : player.getFleet()) {
            if (ship.isAvailable()) {
                activeShip = ship;
                break;
            }
        }
        ICityPlayerProxyJFX proxy = new CityPlayerProxyJFX(player.getHometown(), player, activeShip);
        currentCityProxy = Optional.of(proxy);
    }

}
