package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.graphic.image.impl.XMLImageLoader;
import ch.sahits.game.javafx.control.BaleAmountAlwaysVisible;
import ch.sahits.game.javafx.control.BarrelAmountAlwaysVisible;
import ch.sahits.game.javafx.control.CoinPriceAlwaysVisible;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ETradeType;
import ch.sahits.game.openpatrician.model.building.IAutomatedTrading;
import ch.sahits.game.openpatrician.model.building.ISteward;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.impl.Steward;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.HBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

/**
 * Automatic trading overview.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 8, 2013
 *
 */
@UniquePrototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class AutomaticTradingOverviewDialog extends TabelViewDialog {
	private final ETradeType[] types = new ETradeType[]{ETradeType.NONE,ETradeType.OFFICE_CITY,ETradeType.CITY_OFFICE};
	@Autowired
	@Qualifier("mainScreenXMLImageLoader")
	private XMLImageLoader imageLoader;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;

	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private Locale locale;
    @Autowired
    private MessageSource messageSource;
//    private ResourceBundle resourceBundle = ResourceBundle.getBundle("text.properties");
	
	private final Font font;
	public AutomaticTradingOverviewDialog(Font font, ICityPlayerProxyJFX city) {
		super(font);
		this.city = city;
		this.font = font;
	}
    @PostConstruct
    private void initializeTitle() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.automatic.trading", new Object[]{}, locale.getCurrentLocal()));
    }
	@Override
	protected void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}
	@PostConstruct
	private void initializeModelAndDialog() {
		Table model = createModel();
        if (model != null) {
		    setModel(model);
        }

        final ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());

        HBox box = new HBox();
        box.setSpacing(15);
        box.setLayoutX(2*FRAME_BORDER);
        box.setLayoutY(CLOSE_BTN_Y_POS);

        Text label = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.office.manager", new Object[]{}, locale.getCurrentLocal()));
        label.setFont(font);
        box.getChildren().add(label);

        OpenPatricianLargeWaxButton hireButton = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.hire", new Object[]{}, locale.getCurrentLocal()), font);
        hireButton.textProperty().bind(new HireButtonTextBinding(office));
        hireButton.onActionProperty().bind(new HireActionBinding(office));
        box.getChildren().add(hireButton);

        getContent().add(box);
	}
	private Table createModel() {
		final ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
		ISteward steward = office.getSteward();
		if (steward != null) {
			return createAutomaticTradingModel();
		}
        return null;
	}

	private Table createAutomaticTradingModel() {
		final ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
		Table model = new Table();
		TableHeader header = new TableHeader(10);
		header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal())));
		header.add(new StaticTextTableCell("")); // trading type
		header.add(new StaticTextTableCell("")); // decrease amount
		header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.amount", new Object[]{}, locale.getCurrentLocal())));
		header.add(new StaticTextTableCell("")); // increase amount
//        header.add(new StaticTextTableCell("")); // check
        Image dontMove = imageLoader.getImage("icons/dontMoveOnShipIcon", 96, 48);
        ImageView headerIcon = new ImageView(dontMove);
        header.add(new ControlTableCell(headerIcon));
        header.add(new StaticTextTableCell("")); // min
		header.add(new StaticTextTableCell("")); // decrease price
		header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.price", new Object[]{}, locale.getCurrentLocal())));
		header.add(new StaticTextTableCell("")); // increase price
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.LEFT);
        header.setAligenment(2, HPos.LEFT);
        header.setAligenment(3, HPos.RIGHT);
        header.setAligenment(4, HPos.LEFT);
        header.setAligenment(5, HPos.LEFT);
        header.setAligenment(6, HPos.LEFT);
        header.setAligenment(7, HPos.LEFT);
        header.setAligenment(8, HPos.RIGHT);
        header.setAligenment(9, HPos.LEFT);
		model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.LEFT);
        model.setAligenment(2, HPos.LEFT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.LEFT);
        model.setAligenment(5, HPos.LEFT);
        model.setAligenment(6, HPos.LEFT);
        model.setAligenment(7, HPos.LEFT);
        model.setAligenment(8, HPos.RIGHT);
        model.setAligenment(9, HPos.LEFT);
		model.setColumnWidth(60, 62, 24, 70, 24, 24, 24, 30, 70, 24);
		final IAutomatedTrading autoTrading = office.getOfficeTrading();
		Image minus = imageLoader.getImage("icons/minusIcon");
		Image plus = imageLoader.getImage("icons/plusIcon");
		final Image unchecked = imageLoader.getImage("images/waxseal_24x24");
		final Image checked = imageLoader.getImage("icons/waxseal_checked_icon");
		
		for (final EWare ware : EWare.values()) {
			TableRow row = new TableRow();
			row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));
			
			final ObjectProperty<ETradeType> tradeType = autoTrading.tradingTypeProperty(ware);
			ObjectBinding<Image> iconBinding = new TradingSwitchObjectBinding(tradeType);
			ImageView icon = new ImageView();
			icon.imageProperty().bind(iconBinding);
			Label tradeTypeLbl = new Label("", icon);
            tradeTypeLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    ETradeType nextTradeType = tradeType.get();
                    for (int i = 0; i < types.length; i++) {
                        if (nextTradeType == types[i]) {
                            nextTradeType = types[(i+1)%types.length];
                            break;
                        }
                    }
                    tradeType.set(nextTradeType);
                }
            });
			row.add(new ControlTableCell(tradeTypeLbl));
			
			Label reduceAmountLbl = new Label("", new ImageView(minus));
            reduceAmountLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    autoTrading.updateAmount(ware, -1);
                }
            });
			row.add(new ControlTableCell(reduceAmountLbl));
			
			if (ware.isBarrelSizedWare()) {
				BarrelAmountAlwaysVisible amount = new BarrelAmountAlwaysVisible(font);
				amount.amountProperty().bind(autoTrading.amountProperty(ware).asString());
				row.add(new ControlTableCell(amount));
			} else {
				BaleAmountAlwaysVisible amount = new BaleAmountAlwaysVisible(font);
				amount.amountProperty().bind(autoTrading.amountProperty(ware).asString());
				row.add(new ControlTableCell(amount));
			}
			
			Label increaseAmountLbl = new Label("", new ImageView(plus));
            increaseAmountLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    autoTrading.updateAmount(ware, 1);
                }
            });
			row.add(new ControlTableCell(increaseAmountLbl));
			
			ImageView checkedIcon = new ImageView();
			checkedIcon.imageProperty().bind(new ObjectBinding<Image>() {
				{
					super.bind(autoTrading.movableToShipProperty(ware));
				}
				@Override
				protected Image computeValue() {
					if (autoTrading.isMovableToShip(ware)) {
						return unchecked;
					} else {
						return checked;
					}
				}
			});
			Label checkedLbl = new Label("", checkedIcon);
            checkedLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    autoTrading.setMovableToShip(ware, !autoTrading.isMovableToShip(ware));
                }
            });
			row.add(new ControlTableCell(checkedLbl));

			Text priceTypeText = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.min", new Object[]{}, locale.getCurrentLocal()));
			priceTypeText.setFont(font);
            priceTypeText.visibleProperty().bind(new VisibleTradeTypeBinding(autoTrading.tradingTypeProperty(ware)));
			row.add(new ControlTableCell(priceTypeText));
            Label decreasePriceLbl = new Label("", new ImageView(minus));
            decreasePriceLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    autoTrading.updatePrice(ware, -1);
                }
            });
            decreasePriceLbl.visibleProperty().bind(new VisibleTradeTypeBinding(autoTrading.tradingTypeProperty(ware)));
            row.add(new ControlTableCell(decreasePriceLbl));

			CoinPriceAlwaysVisible price = new CoinPriceAlwaysVisible(font);
			price.amountProperty().bind(autoTrading.priceProperty(ware).asString());
            price.visibleProperty().bind(new VisibleTradeTypeBinding(autoTrading.tradingTypeProperty(ware)));
			row.add(new ControlTableCell(price));
			
			Label increasePriceLbl = new Label("", new ImageView(plus));
            increasePriceLbl.setOnMouseReleased(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                        autoTrading.updatePrice(ware, 1);
                }
            });
            increasePriceLbl.visibleProperty().bind(new VisibleTradeTypeBinding(autoTrading.tradingTypeProperty(ware)));
			row.add(new ControlTableCell(increasePriceLbl));
			
			model.add(row);
		}

		return model;
	}

    /**
     * Image binding based on the trade type.
     */
	private class TradingSwitchObjectBinding extends ObjectBinding<Image> {

		private final ObjectProperty<ETradeType> boundTradeType;
		TradingSwitchObjectBinding(ObjectProperty<ETradeType> tradeTypeProperty) {
			super.bind(tradeTypeProperty);
			boundTradeType = tradeTypeProperty;
		}
		@Override
		protected Image computeValue() {
			switch (boundTradeType.get()) {
			case NONE:
				return imageLoader.getImage("images/waxseal", 62, 24);
			case CITY_OFFICE:
				return imageLoader.getImage("icons/townOutIcon", 62, 24);
			case OFFICE_CITY:
				return imageLoader.getImage("icons/townInIcon", 62, 24);
			default:
				throw new IllegalArgumentException("Illegal trading type: "+boundTradeType.get());
			}
		}
		
	}

    /**
     * Visibility binding to evaluate to true, when the trade type is not ETradeType#NONE.
     */
    private class VisibleTradeTypeBinding extends BooleanBinding {
        private final ObjectProperty<ETradeType> boundTradeType;

        private VisibleTradeTypeBinding(ObjectProperty<ETradeType> boundTradeType) {
            this.boundTradeType = boundTradeType;
            super.bind(boundTradeType);
        }

        @Override
        protected boolean computeValue() {
            return boundTradeType.get() != ETradeType.NONE;
        }
    }

    private class HireButtonTextBinding extends StringBinding {
        private final ITradingOffice office;

        private HireButtonTextBinding(ITradingOffice office) {
            this.office = office;
            super.bind(office.storageManagerPresentBinding());
        }

        @Override
        protected String computeValue() {
            if (office.storageManagerPresentBinding().get()) {
                return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.fire", new Object[]{}, locale.getCurrentLocal());
            } else {
                return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.AutomaticTradingOverviewDialog.hire", new Object[]{}, locale.getCurrentLocal());
            }
        }
    }
    private class HireActionBinding extends ObjectBinding<EventHandler<MouseEvent>> {
        private final ITradingOffice office;

        private HireActionBinding(ITradingOffice office) {
            this.office = office;
            super.bind(office.storageManagerPresentBinding());
        }

        @Override
        protected EventHandler<MouseEvent> computeValue() {
            if (office.storageManagerPresentBinding().get()) {
                return new EventHandler<MouseEvent>() {

                    @Override
                    public void handle(MouseEvent event) {
                        office.setSteward(null);
                        Table newModel = new Table();
                        setModel(newModel);
                    }

                };
            } else {
                return new EventHandler<MouseEvent>() {

                    @Override
                    public void handle(MouseEvent event) {
                        office.setSteward(new Steward());
                        Table newModel = createAutomaticTradingModel();
                        setModel(newModel);
                    }

                };
            }
        }
    }


}
