package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.graphic.display.dialog.action.BuyJFXProxyAction;
import ch.sahits.game.graphic.display.dialog.action.SellJFXProxyAction;
import ch.sahits.game.graphic.display.dialog.util.EDialogType;
import ch.sahits.game.graphic.display.dialog.util.ITransferableJFX;
import ch.sahits.game.javafx.control.BaleAmount;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.CoinPrice;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.TransferUtil;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.NumberBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.input.MouseEvent;
import javafx.scene.text.Font;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

public abstract class BaseTradeDialog extends TabelViewDialog implements ITransferableJFX {
    private ObjectProperty<ETransferAmount> movableAmount = new SimpleObjectProperty<>(ETransferAmount.ONE);

	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
    @Autowired
    private TransferUtil transferUtil;
	@Autowired
	private Locale locale;
	@Autowired
	private ModelTranslations translator;
	@Autowired
	private MessageSource messageSource;

	private EDialogType type;
	private Font font;

	public BaseTradeDialog(Font font, ICityPlayerProxyJFX city, EDialogType type) {
		super(font);
		this.city = city;
		this.type = type;
		this.font = font;
	}
	/**
	 * Provide a group for the sub title.
	 * @param font
	 * @param city
	 * @return
	 */
	protected abstract Group createSubTitle(Font font, ICityPlayerProxyJFX city);
	@PostConstruct
	private void initializeModelAndDialog() {
		Group subTitle = createSubTitle(font, city);
		subTitle.setLayoutX(2*FRAME_BORDER);
		subTitle.setLayoutY(80);
		getContent().add(subTitle);

		setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.title", new Object[]{}, locale.getCurrentLocal()));

		OpenPatricianSmallWaxButton btn1 = new OpenPatricianSmallWaxButton("1", font);
		btn1.setOnAction(new EventHandler<MouseEvent>() {
			@Override
			public void handle(MouseEvent arg0) {
				movableAmount.setValue(ETransferAmount.ONE);
			}
		});
		OpenPatricianSmallWaxButton btn5 = new OpenPatricianSmallWaxButton("5", font);
		btn5.setOnAction(new EventHandler<MouseEvent>() {
			@Override
			public void handle(MouseEvent arg0) {
				movableAmount.setValue(ETransferAmount.FIVE);
			}
		});
		String max = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.max", new Object[]{}, locale.getCurrentLocal());
		OpenPatricianSmallWaxButton btnMax = new OpenPatricianSmallWaxButton(max, font);
		btnMax.setOnAction(new EventHandler<MouseEvent>() {
			@Override
			public void handle(MouseEvent arg0) {
				movableAmount.setValue(ETransferAmount.MAX);
			}
		});
		addButtomControlButtons(btn1, btn5, btnMax);
		Table model = createModel(city, font);
		setModel(model);
	}


	private Table createModel(ICityPlayerProxyJFX cityProxy, Font font) {
    	Table model = new Table();

    	TableHeader header = new TableHeader(6);
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(getTradeFromDestination()));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.buy", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.sell", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(getTradeToDestination()));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.avg-price", new Object[]{}, locale.getCurrentLocal())));
        header.setAligenment(0, HPos.RIGHT);
        header.setAligenment(1, HPos.RIGHT);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.RIGHT);
        header.setAligenment(5, HPos.RIGHT);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.LEFT);
        model.setAligenment(3, HPos.LEFT);
        model.setAligenment(4, HPos.RIGHT);
        model.setAligenment(5, HPos.RIGHT);
    	model.setColumnWidth(100, 70, 62, 62, 70, 70);
    	ICity city = cityProxy.getCity();
    	ITradingOffice office = cityProxy.getPlayer().findTradingOffice(city);
    	IShip ship = cityProxy.getActiveShip();
       	for (final EWare ware : EWare.values()) {
    			TableRow row = new TableRow();
    			row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));
    			IntegerProperty amountAvailableProp = getAvailableAmountProperty(city, ware, office);
    			IntegerBinding amountToTransfer = getAmountProperty(city.getWare(ware).amountProperty());
    			if (ware.isBarrelSizedWare()) {
    				BarrelAmount barrelAmount = new BarrelAmount(font);
    				barrelAmount.amountProperty().bind(amountAvailableProp.asString());
    				row.add(new ControlTableCell(barrelAmount));
    			} else {
    				BaleAmount baleAmount = new BaleAmount(font);
    				baleAmount.amountProperty().bind(amountAvailableProp.asString());
    				row.add(new ControlTableCell(baleAmount));
    			}
    			OpenPatricianSmallWaxButton buyBtn = new OpenPatricianSmallWaxButton("",font);
    			buyBtn.textProperty().bind(buyPrice(ware, amountAvailableProp, amountToTransfer));
    			buyBtn.setOnAction(new EventHandler<MouseEvent>() {

					@Override
					public void handle(MouseEvent arg0) {
						new BuyJFXProxyAction(ware, BaseTradeDialog.this).run();
					}
				});
    			row.add(new ControlTableCell(buyBtn));
    			OpenPatricianSmallWaxButton sellBtn = new OpenPatricianSmallWaxButton("", font);
    			sellBtn.textProperty().bind(sellPrice(ware, amountAvailableProp, amountToTransfer));
    			sellBtn.setOnAction(new EventHandler<MouseEvent>() {

					@Override
					public void handle(MouseEvent arg0) {
						new SellJFXProxyAction(ware, BaseTradeDialog.this).run();
						
					}
				});
    			row.add(new ControlTableCell(sellBtn));
    			IntegerProperty storedAmountProperty = getStoredAmountProperty(ware, ship, office);
    			if (ware.isBarrelSizedWare()) {
    				BarrelAmount barrelAmount = new BarrelAmount(font);
    				barrelAmount.amountProperty().bind(storedAmountProperty.asString());
    				row.add(new ControlTableCell(barrelAmount));
    			} else {
       				BaleAmount baleAmount = new BaleAmount(font);
    				baleAmount.amountProperty().bind(storedAmountProperty.asString());
    				row.add(new ControlTableCell(baleAmount));
 	    		}
    			CoinPrice coinPrice = new CoinPrice(font);
    			AmountablePrice<IWare> amountable = getStoredAmountablePrice(ware, ship, office);
    			final NumberBinding avgPriceProperty = amountable.avgPriceProperty();
    			StringBinding avgPriceStringBinding = new StringBinding() {
    				{
    					super.bind(avgPriceProperty);
    				}
					
					@Override
					protected String computeValue() {
						return String.valueOf((int)Math.rint(avgPriceProperty.doubleValue()));
					}
				}; 
    			coinPrice.amountProperty().bind(avgPriceStringBinding);
    			row.add(new ControlTableCell(coinPrice));
    			model.add(row);
    		}
		return model;
	}
	/**
	 * Trade from the city.
	 * @return
	 */
	protected String getTradeFromDestination() {
		return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.city", new Object[]{}, locale.getCurrentLocal());
	}
	/**
	 * Trade to the ship.
	 * @return
	 */
	protected String getTradeToDestination() {
		return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.ship", new Object[]{}, locale.getCurrentLocal());
	}
    /**
     * Retrieve the stored amount
     * @param ware Reference to the ware
     * @param ship Reference to the ship
     * @param office Reference to the trading office
     * @return
     */
	private IntegerProperty getStoredAmountProperty(IWare ware, IShip ship, ITradingOffice office){
		AmountablePrice<IWare> amounable = getStoredAmountablePrice(ware, ship,
				office);
		return amounable.amountProperty(); // differ between ship and convoi
	}
	/**
	 * Retrieve the amountable price of a ware from the ship or the warehouse
	 * @param ware
	 * @param ship
	 * @param office
	 * @return
	 */
	private AmountablePrice<IWare> getStoredAmountablePrice(IWare ware, IShip ship,
			ITradingOffice office) {
		AmountablePrice<IWare> amounable;
		switch (type) {
		case CITY_TO_SHIP:
		case STORAGE_TO_SHIP: // stored here means stored on the ship
			amounable = ship.getWare(ware);
			break;
		case CITY_TO_STORAGE:
			amounable = office.getWare(ware);
			break;
		default:
			// This should only occur for weapons which have their own dialog
			throw new IllegalStateException("Dialog type "+type+" not handled");
		}
		return amounable;
	}
	private IntegerProperty getAvailableAmountProperty(ICity city, IWare ware, ITradingOffice office) {
		AmountablePrice<IWare> available;
		switch (type) {
		case CITY_TO_SHIP:
		case CITY_TO_STORAGE:
			available = city.getWare(ware);
			break;
		default:
			available = office.getWare(ware);
			break;
		}
		return available.amountProperty();
	}
	/**
	 * Create the binding for the buy price for the indicated ware.
	 * @param ware
	 * @param availableAmount
	 * @param amountToBuy
	 * @return
	 */
	private StringBinding buyPrice(final IWare ware, final IntegerProperty availableAmount, final IntegerBinding amountToBuy) {
		return new StringBinding() {
			{
				super.bind(availableAmount, amountToBuy);
			}
			
			@Override
			protected String computeValue() {
				switch (type) {
				case CITY_TO_SHIP:
				case CITY_TO_STORAGE:
					if (availableAmount.get() > 0){
						return String.valueOf(ware.buyPrice(availableAmount, amountToBuy));
					} else {
						return "0"; // cannot buy anything if nothing is there
					}
				default:
					return "<";
				}
			}
		};
	}
	/**
	 * Create a binding for the sell price for the indicated ware.
	 * @param ware
	 * @param availableAmount
	 * @param amountToSell
	 * @return
	 */
	private StringBinding  sellPrice(final IWare ware, final IntegerProperty availableAmount,final IntegerBinding amountToSell) {
		return new StringBinding() {
			{
				super.bind(availableAmount, amountToSell);
			}
			@Override
			protected String computeValue() {
				switch (type) {
				case CITY_TO_SHIP:
				case CITY_TO_STORAGE:
					if (movableAmount.get()==ETransferAmount.MAX){
						return String.valueOf(ware.getMaxValueSell());
					} else if (amountToSell.get() > 0){
						return String.valueOf(ware.sellPrice(availableAmount, amountToSell));
					} else {
						return String.valueOf(ware.getMaxValueSell());
					}
				default:
					return ">";
				}
			}
		};
	}
	



	/**
	 * Bind the amount that can be transferred on the maximal available amount
	 * as well as the selected transfer amount.
	 * @param maxAmount
	 * @return
	 */
	private IntegerBinding getAmountProperty(final IntegerProperty maxAmount) {
		return new IntegerBinding() {
			{
				super.bind(movableAmount, maxAmount);
			}
			
			@Override
			protected int computeValue() {
				return transferUtil.calculateAvailableAmount(movableAmount, maxAmount.get());
			}
		};
	}
	@Override
	public ICityPlayerProxyJFX getCityPlayerProxy() {
		return city;
	}
	@Override
	public EDialogType getDialogType() {
		return type;
	}
	@Override
	public int getAmount(int availableAmountCity) {
		return transferUtil.calculateAvailableAmount(movableAmount, availableAmountCity);
	}
	@Override
	public ETransferAmount getMovableAmount() {
		return movableAmount.get();
	}

}
