package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.javafx.control.BaleAmount;
import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.PlaceHolder;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.city.BuildingProduction;
import ch.sahits.game.openpatrician.model.city.CityProduction;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.property.IntegerProperty;
import javafx.geometry.HPos;
import javafx.scene.Group;
import javafx.scene.layout.StackPane;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

/**
 * Dialog listing the production and consumtion of wares in the city.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 7, 2013
 *
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class CityProductionConsumptionDialog extends TabelViewDialog {

	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
	
	private final Font font;
	@Autowired
	private CityProduction cityProduction;
	@Autowired
	private BuildingProduction buildingProduction;
	@Autowired
//	@Qualifier("populationConsueProperty")
	private PopulationConsume consumer;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
	@Autowired
	private Locale locale;
	@Autowired
	private ModelTranslations translator;
	@Autowired
	private MessageSource messageSource;


	public CityProductionConsumptionDialog(Font font, ICityPlayerProxyJFX city) {
		super(font);
		this.city = city;
		this.font = font;

	}
	@PostConstruct
	private void initializeModelAndDialog() {
		setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.prodCon", new Object[]{}, locale.getCurrentLocal()));
		Group subTitle = createSubTitle();
		subTitle.setLayoutX(2*FRAME_BORDER);
		subTitle.setLayoutY(80);
		getContent().add(subTitle);
		Table model = createModel();
		setModel(model);
	}

	private Group createSubTitle() {
		Group g = new Group();
		g.setManaged(false);
		Text text = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.weekly", new Object[]{}, locale.getCurrentLocal()));
		text.setFont(font);
		StackPane titlePane = new StackPane();
		titlePane.getChildren().addAll(new PlaceHolder(WIDTH, 1), text);

		g.getChildren().add(titlePane);
		return g;
	}

	private Table createModel() {
    	Table model = new Table();

    	TableHeader header = new TableHeader(7);
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.ware", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.stock", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.city", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.shops", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.pop", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.shops", new Object[]{}, locale.getCurrentLocal())));
    	header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.CityProductionConsumptionDialog.tot", new Object[]{}, locale.getCurrentLocal())));
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.RIGHT);
        model.setAligenment(2, HPos.RIGHT);
        model.setAligenment(3, HPos.RIGHT);
        model.setAligenment(4, HPos.RIGHT);
        model.setAligenment(5, HPos.RIGHT);
        model.setAligenment(6, HPos.RIGHT);
    	model.setColumnWidth(70, 60, 60, 60, 60, 60, 60);
    	
       	for (final EWare ware : EWare.values()) {
			TableRow row = new TableRow();
			
			row.add(new StaticTextTableCell(translator.getLocalDisplayName(ware)));
			
			IntegerProperty amountAvailableProp = city.getCity().getWare(ware).amountProperty();
			if (ware.isBarrelSizedWare()) {
				BarrelAmount barrelAmount = new BarrelAmount(font);
				barrelAmount.amountProperty().bind(amountAvailableProp.asString());
				row.add(new ControlTableCell(barrelAmount));
			} else {
				BaleAmount baleAmount = new BaleAmount(font);
				baleAmount.amountProperty().bind(amountAvailableProp.asString());
				row.add(new ControlTableCell(baleAmount));
			}
			
			int producedAmount = calculateProducedWares(ware);
			addAmountCell(ware, row, producedAmount);
			
			int chandlerProduction = buildingProduction.getTotalProduction(ware, city.getCity());
			addAmountCell(ware, row, chandlerProduction);
			
			int popConsumption = computeCitizenConsumtion(ware);
			addAmountCell(ware, row, popConsumption);

			int consumeShops = buildingProduction.getTotalConsumtion(ware, city.getCity());
			addAmountCell(ware, row, consumeShops);
			
			int consumeTot = consumeShops;
			consumeTot += computeCitizenConsumtion(ware);
			addAmountCell(ware, row, consumeTot);
			
			model.add(row);

       	} // end for row

    	
    	return model;
	}
	private void addAmountCell(final EWare ware, TableRow row,
			int popConsumption) {
		if (ware.isBarrelSizedWare()) {
			BarrelAmount barrelAmount = new BarrelAmount(font);
			barrelAmount.setAmount(popConsumption);
			row.add(new ControlTableCell(barrelAmount));
		} else {
			BaleAmount baleAmount = new BaleAmount(font);
			baleAmount.setAmount(popConsumption);
			row.add(new ControlTableCell(baleAmount));
		}
	}
	
	private int calculateProducedWares(IWare ware) {
		IWare[] effective = city.getCity().getEffectiveProduction();
		IWare[] ineffective = city.getCity().getIneffectiveProduction();
		int amount=0;
		if (contains(effective, ware)){
			amount = cityProduction.getEfficientProduction(ware);
		} else if (contains(ineffective, ware)){
			amount = cityProduction.getInefficientProduction(ware);
		}
		return amount;
	}
	/**
	 * check if the array contains the item
	 * @param array
	 * @param item
	 * @return
	 */
	private boolean contains(IWare[] array, IWare item){
		for (IWare ware : array) {
			if (ware==item) {
				return true;
			}
		}
		return false;
	}
	/**
	 * Compute the amount that all the citizens consume
	 * @param ware Reference to the ware
	 * @return amount all the citizens consume of that ware
	 */
	private int computeCitizenConsumtion(IWare ware) {
		double amount = consumer.getNeed(ware, EPopulationClass.POOR, city.getCity().getPopulation(EPopulationClass.POOR));
		amount += consumer.getNeed(ware, EPopulationClass.MEDIUM, city.getCity().getPopulation(EPopulationClass.MEDIUM));
		amount += consumer.getNeed(ware, EPopulationClass.RICH, city.getCity().getPopulation(EPopulationClass.RICH));
		int value = (int) Math.rint(amount);
		return value;
	}
	@Override
	protected void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}

}
