package ch.sahits.game.graphic.display.dialog;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.graphic.javafx.display.MainGameView;
import ch.sahits.game.javafx.control.PaginationV2;
import ch.sahits.game.javafx.control.PaginationV2Builder;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.personal.IMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.TextParser;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.geometry.Insets;
import javafx.scene.Node;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * Dialog presenting personal data.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Dec 7, 2013
 *
 */
@UniquePrototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.SINGLETON_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class HistoryStandaloneDialog extends CloseButtonDialog {
	private final Logger logger = LogManager.getLogger(getClass());

	private Font font;
	/** Reference to the city view model */
	protected final ICityPlayerProxyJFX city;
	private final String historyResource;
	@Autowired
	private MessageSource messageSource;
	@Autowired
	private Locale locale;
	@Autowired
	@Qualifier("resourceReference")
	private MessageSource resources;
	@Autowired
	private TextParser textParser;
	@Autowired
	private IFontLoader fontLoader;

	@Autowired
	private ModelTranslations modelTranslator;

	public HistoryStandaloneDialog(Font font, ICityPlayerProxyJFX city, String historyResource) {
		super(font);
		this.font = font;
		this.city = city;
		this.historyResource = historyResource;
	}
	@PostConstruct
	private void initializeDialog() {
		final ch.sahits.game.openpatrician.util.model.Text historyText = getHistoryText();
		Insets insets = new Insets(FRAME_BORDER, FRAME_BORDER, FRAME_BORDER, FRAME_BORDER * 2);
		Font headerFont = fontLoader.createDefaultFont((int)font.getSize() + 6); // bold font not available
		PaginationV2 pagination = PaginationV2Builder.create()
				.backButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.back", new Object[]{}, locale.getCurrentLocal()))
				.nextButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.next", new Object[]{}, locale.getCurrentLocal()))
				.text(historyText)
				.contentMaxWidth(WRAPPING_WIDTH)
				.contentMaxHeight(CloseButtonDialog.CLOSE_BTN_Y_POS - 4 * FRAME_BORDER)
				.padding(insets)
				.navigationLabelFont(font)
				.headerFont(headerFont)
				.paragraphFont(font)
				.centeredFont(font)
				.firstLastPageAction(event -> {
							MainGameView mainGameView = (MainGameView) getParent();
							mainGameView.closeDialog();
						}
				)
				.build();
		pagination.setLayoutY(FRAME_BORDER * 2);

//		getContent().remove(getCloseButton()); // todo: andi 9/26/15: if the close button is removed or hidden the content no longer renders

		getContent().add(pagination);
	}

	/**
	 * Retrieve the history text.
	 * @return
	 */
	private ch.sahits.game.openpatrician.util.model.Text getHistoryText() {
		String resourceName = resources.getMessage(historyResource, new Object[0], locale.getCurrentLocal());
		URL url = getClass().getClassLoader().getResource(resourceName);
		try {
			InputStream is= url.openStream();
			return textParser.parse(is);
		} catch (IOException e) {
			logger.warn("History text could not be found: " + url.toExternalForm());
			return new ch.sahits.game.openpatrician.util.model.Text();
		}
	}



	private void createAndAddNewText(VBox box, String s) {
		Text spouseName = new Text(s);
		spouseName.setFont(font);
		box.getChildren().add(spouseName);
	}

	private Node createEmptyNode() {
		Text text = new Text("");
		text.setFont(font);
		return text;
	}

	private String getReputationText(IReputation reputation) {
		int repDiff = reputation.getReputationDifference();
		StringBuilder sb = new StringBuilder();
		if (repDiff<0){
			// reputation has sunk
			if (repDiff<=-10 && repDiff>-25){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff>-10){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.sunkenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else if (repDiff>0){
			// reputation has risen
			if (repDiff>=10 && repDiff<25){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep1", new Object[]{}, locale.getCurrentLocal()));
			} else if (repDiff<10){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep2", new Object[]{}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.risenRep3", new Object[]{}, locale.getCurrentLocal()));
			}
		} else {
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.unchangedRep", new Object[]{}, locale.getCurrentLocal()));
		}
		return sb.toString();
	}
	/**
	 * Get the two lines of text for career advice.
	 * @param player
	 * @return
	 */
	private List<String> getCareerAdvice(IPlayer player, IReputation reputation){
		ESocialRank rank = player.getRank();
		ArrayList<String> lines = new ArrayList<>();
		if (rank!=ESocialRank.ALDERMAN && rank!=ESocialRank.MAYOR){
			// draw string what is needed
			ESocialRank nextRank = rank.getNextRank();
			boolean needFortune = false;
			if (player.getCompany().getCompanyValue()<nextRank.getCompanyValue()){
				needFortune=true;
			}
			boolean needReputation=false;
			if (reputation.getPopularity()<nextRank.getSocialReputation()){
				needReputation=true;
			}
			StringBuilder sb = new StringBuilder();
			if (needFortune && needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needFortune){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice2_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			} else if (needReputation){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice3_1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
				lines.add(sb.toString());
				sb = new StringBuilder();
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.careerAdvice1_2", new Object[]{}, locale.getCurrentLocal()));
				lines.add(sb.toString());
			}
		}
		return lines;
	}

	public List<String> getSpouseReputation(IPlayer player, ISpouseData spouse) {
		ArrayList<String> lines = new ArrayList<>();
		StringBuilder sb = new StringBuilder();
		if (city.getCity()==spouse.getBirthPlace()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else if (city.getCity()==player.getCompany().getHomeTown()){
			// TODO consider the spouses popularity and relations
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		} else {
			if (spouse.isMale()){
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep1", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			} else {
				sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep2", new Object[]{city.getCity().getName()}, locale.getCurrentLocal()));
			}
			lines.add(sb.toString());
			sb = new StringBuilder();
			lines.add(sb.toString());
			sb.append(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.spouseRep3", new Object[]{}, locale.getCurrentLocal()));
		}
		return lines;
	}

	private String getUnderworldReputation(IPlayer player) {
		ICareer careerLevel = player.getCareerLevel();
		String s;
		if (careerLevel instanceof IMilitantCareer){
			if (((EMilitantCareer)careerLevel)!=EMilitantCareer.BOISTEROUS &&
					((EMilitantCareer)careerLevel)!=EMilitantCareer.BOLD ){
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep1", new Object[]{}, locale.getCurrentLocal());
			} else {
				s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
			}
		} else {
			s = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.underworldRep2", new Object[]{}, locale.getCurrentLocal());
		}
		return s;
	}

	private String getPrefix(ISpouseData spouseData){
		if (spouseData.isMale()){
			return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.malePrefix", new Object[]{}, locale.getCurrentLocal());
		} else {
			return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.PersonalDialog.femalePrefix", new Object[]{}, locale.getCurrentLocal());
		}
	}

	@Override
	protected void executeOnCloseButtonClicked() {
		clientEventBus.post(new NoticeBoardClose());
		super.executeOnCloseButtonClicked();
	}

}
