package ch.sahits.game.graphic.display.dialog.shipyard;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.graphic.image.impl.XMLImageLoader;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.control.PaginationV2;
import ch.sahits.game.javafx.control.PaginationV2Builder;
import ch.sahits.game.javafx.model.ECellConstraint;
import ch.sahits.game.javafx.model.ITableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.javafx.util.JavaFXUtils;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.TextParser;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.ObjectBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.VPos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 14, 2013
 */
public abstract class BaseShipyardDialog extends CloseButtonDialog {
    private final Logger logger = LogManager.getLogger(getClass());
    /** Reference to the city view model */
    protected final ICityPlayerProxyJFX city;
    protected final Font font;
    private IntegerProperty currentShipTypeIndex;
    protected final EShipType[] shipTypes;
    @Autowired
    private MessageSource messageSource;
    @Autowired
    private Locale locale;
    @Autowired
    @Qualifier("resourceReference")
    private MessageSource resources;
    @Autowired
    private TextParser textParser;
    @Autowired
    @Qualifier("mainScreenXMLImageLoader")
    private XMLImageLoader imageLoader;
    @Autowired
    private JavaFXUtils fxUtils;
    @Autowired
    private IFontLoader fontLoader;

    private ECellConstraint cellConstraint;
    private int numberOfColumns;
    protected BooleanBinding enablePreviousNext;
    protected int mainTableYPosition;

    public BaseShipyardDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
        currentShipTypeIndex = new SimpleIntegerProperty(this,"currentShipTypeIndex", 0);
        IShipyard shipyard = city.getCity().getCityState().getShipyardState();
        shipTypes = shipyard.getBuildableShipTypes();
        enablePreviousNext = new BooleanBinding() {
            @Override
            protected boolean computeValue() {
                return true;
            }
        };
        mainTableYPosition = 250 + 36;
    }
    @PostConstruct
    private void initializeComponents() {
        if (hasShips()) {
            ImageView sideView = new ImageView();
            sideView.imageProperty().bind(shipSideImageBinding());
            sideView.setLayoutY(50);
            sideView.setLayoutX(30);
            final EventHandler<? super MouseEvent> historyDisplayHandler = createHistoryDisplayHandler();
            sideView.setOnMouseReleased(historyDisplayHandler);
            ImageView frontView = new ImageView();
            frontView.imageProperty().bind(shipFrontImageBinding());
            frontView.setLayoutY(50);
            frontView.setLayoutX(289);
            frontView.setOnMouseReleased(historyDisplayHandler);
            getContent().addAll(sideView, frontView);
            Table topTable = createTopTable();
            GridPane topTablePane = new GridPane();
            topTablePane.setLayoutX(50);
            topTablePane.setLayoutY(250);
            RowConstraints rowConstraint = new RowConstraints(48);
            rowConstraint.setValignment(VPos.TOP);
            topTablePane.getRowConstraints().add(rowConstraint);
            for (TableRow row : topTable) {
                for (int i = 0; i < topTable.getNumberOfColumns(); i++) {
                    int colWidth = topTable.getColumnWidth(i);
                    ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
                    colConstraint.setHalignment(topTable.getAligenment(i));
                    topTablePane.getColumnConstraints().add(colConstraint);
                    ITableCell cell = row.get(i);
                    final String textStyleClass = "tableCell";
                    fxUtils.addCellToGridPane(font, topTablePane, -1, null, i, cell, topTable.getAligenment(i), textStyleClass);
                }
            }
            Table mainTable = createMainTable();
            GridPane mainTablePane = new GridPane();
            mainTablePane.setLayoutX(2 * FRAME_BORDER);
            mainTablePane.setLayoutY(mainTableYPosition);
            rowConstraint = getRowConstraints();
            mainTablePane.getRowConstraints().add(rowConstraint);
            // Header
            TableHeader header = mainTable.getHeader();
            numberOfColumns = mainTable.getNumberOfColumns();
            int currentCol = 0;
            for (int i = 0; i < numberOfColumns && header.size() > 0; i++) {
                int colWidth = mainTable.getColumnWidth(i);
                ColumnConstraints colConstraint = new ColumnConstraints(colWidth);
                colConstraint.setHalignment(header.getAligenment(i));
                mainTablePane.getColumnConstraints().add(colConstraint);
                ITableCell cell = header.get(i);
                final String textStyleClass = "tableHeader";
                fxUtils.addCellToGridPane(font, mainTablePane, -1, null, currentCol++, cell, header.getAligenment(i), textStyleClass);
                // jump over spanned columns
                cellConstraint = header.getCellConstraint(cell);
                if (cellConstraint != null) {
                    switch (cellConstraint) {
                        case COLSPAN2:
                            numberOfColumns--;
                            currentCol++;
                            break;
                        case COLSPAN4:
                            numberOfColumns -= 3;
                            currentCol += 3;
                            break;
                        default:
                            throw new RuntimeException("Unhandled case");
                    }
                }
            }
            // Table
            for (int rowNum = 0; rowNum < mainTable.getNumberOfRows(); rowNum++) {
                TableRow row = mainTable.get(rowNum);
                mainTablePane.getRowConstraints().add(rowConstraint);
                final int iterationLimit = Math.min(mainTable.getNumberOfColumns(), row.size());
                for (int col = 0; col < iterationLimit; col++) {

                    ITableCell cell = row.get(col);
                    final HPos hAlignment = mainTable.getAligenment(col);
                    final String textStyleClass = "tableCell";
                    fxUtils.addCellToGridPane(font, mainTablePane, rowNum, row, col, cell, hAlignment, textStyleClass);
                }
            }
            getContent().addAll(topTablePane, mainTablePane);

            Group footerText = createFooterText();
            footerText.setLayoutX(50);
            footerText.setLayoutY(250 + 36 + 7 * 24);

            final int actionButtonX = (WIDTH - 124) / 2;

            final OpenPatricianSmallWaxButton previous = new OpenPatricianSmallWaxButton("<", font);
            previous.setLayoutX(actionButtonX - 62 - 4);
            previous.setLayoutY(CLOSE_BTN_Y_POS - 48);
            previous.setOnAction(createPreviousAction());
            final OpenPatricianSmallWaxButton next = new OpenPatricianSmallWaxButton(">", font);
            next.setLayoutX(actionButtonX + 124 + 4);
            next.setLayoutY(CLOSE_BTN_Y_POS - 48);
            next.setOnAction(createNextAction());

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(getActionText(), font);
            action.setOnAction(getAction());
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            BooleanBinding actionEnabled = actionEnabledBinding();
            action.setDisable(!actionEnabled.get());
            actionEnabled.addListener(new ChangeListener<Boolean>() {
                @Override
                public void changed(ObservableValue<? extends Boolean> observableValue,
                                    Boolean oldValue, Boolean newValue) {
                    action.setDisable(!newValue);
                }
            });

            enablePreviousNext.addListener(new ChangeListener<Boolean>() {
                @Override
                public void changed(ObservableValue<? extends Boolean> observableValue,
                                    Boolean oldVBoolean, Boolean newValue) {
                    next.setDisable(!newValue);
                    previous.setDisable(!newValue);
                }
            });

            getContent().addAll(footerText, previous, next, action);
        }  else {
            // no ships
            Text text = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseShipyardDialog.noShip", new Object[]{}, locale.getCurrentLocal()));
            text.setFont(font);
            text.setLayoutX(50);
            text.setLayoutY(250);
            getContent().addAll(text);
        }
    }

    private EventHandler<? super MouseEvent> createHistoryDisplayHandler() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                final ch.sahits.game.openpatrician.util.model.Text historyText = getHistoryText(getCurrentShipType());
                Insets insets = new Insets(FRAME_BORDER, FRAME_BORDER, FRAME_BORDER, FRAME_BORDER * 2);
                Font headerFont = fontLoader.createDefaultFont((int)getFont().getSize() + 6); // bold font not available
                final List<Node> firstPageContent = new ArrayList<>(getContent().size());
                for (Node node : getContent()) {
                    firstPageContent.add(node);
                }
                PaginationV2 pagination = PaginationV2Builder.create()
                        .backButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.back", new Object[]{}, locale.getCurrentLocal()))
                        .nextButtonLabel(messageSource.getMessage("ch.sahits.game.graphic.display.scene.CreditsScene.next", new Object[]{}, locale.getCurrentLocal()))
                        .text(historyText)
                        .contentMaxWidth(WRAPPING_WIDTH)
                        .contentMaxHeight(CLOSE_BTN_Y_POS - 4 * FRAME_BORDER)
                        .padding(insets)
                        .navigationLabelFont(getFont())
                        .headerFont(headerFont)
                        .paragraphFont(getFont())
                        .centeredFont(getFont())
                        .firstLastPageAction(new ReplaceFirstPage(firstPageContent))
                        .build();
                pagination.setLayoutY(FRAME_BORDER * 2);
                replaceContent(pagination);
                System.out.println("Clicked on ship to display history");
            }
        };
    }
    /**
     * Retrieve the history text.
     * @return
     */
    private ch.sahits.game.openpatrician.util.model.Text getHistoryText(EShipType type) {
        String resourceName = resources.getMessage(getHistoryResourceName(type), new Object[0], locale.getCurrentLocal());
        URL url = getClass().getClassLoader().getResource(resourceName);
        try {
            InputStream is= url.openStream();
            return textParser.parse(is);
        } catch (IOException e) {
            logger.warn("History text could not be found: " + url.toExternalForm());
            return new ch.sahits.game.openpatrician.util.model.Text();
        }
    }

    private String getHistoryResourceName(EShipType type) {
        switch (type) {
            case SNAIKKA:
                return "history.snaikka";
            case CRAYER:
                return "history.crayer";
            case COG:
                return "history.cog";
            case HOLK:
                return "history.holk";
            default:
                throw new IllegalStateException("The ship type "+type+" is not handled.");
        }
    }

    protected RowConstraints getRowConstraints() {
        RowConstraints rowConstraint;
        rowConstraint = new RowConstraints(24);
        return rowConstraint;
    }

    /**
     * Move the selection to the next index
     * @return .
     */
    protected EventHandler<MouseEvent> createNextAction() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
               if (currentShipTypeIndex.get() == shipTypes.length - 1) {
                   currentShipTypeIndex.set(0);
               } else {
                   currentShipTypeIndex.set(currentShipTypeIndex.get() + 1);
               }
            }
        };
    }

    /**
     * move the selection to the previous index.
     * @return  .
     */
    protected EventHandler<MouseEvent> createPreviousAction() {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                if (currentShipTypeIndex.get() == 0) {
                    currentShipTypeIndex.set(shipTypes.length - 1);
                } else {
                    currentShipTypeIndex.set(currentShipTypeIndex.get() - 1);
                }
            }
        };
    }

    /**
     * Define the action that is executed on the action button.
     * @return action that is executed on the button
     */
    protected abstract EventHandler<MouseEvent> getAction();

    /**
     * Label of the action button
     * @return label on the action button
     */
    protected abstract String getActionText();

    /**
     * Create the group for the footer text;
     * @return bottom footer text group
     */
    protected abstract Group createFooterText();

    /**
     * Create a table representing the main information
     * @return Table model for the main part
     */
    protected abstract Table createMainTable();


    /**
     * Create a table representing the header information
     * @return Table model used for the top line
     */
    protected abstract Table createTopTable();

    /**
     * Boolean binding defining if the action button is enabled.
     * @return  .
     */
    protected abstract BooleanBinding actionEnabledBinding();

    /**
     * Binding for the appropriate side image of the ship.
     * @return .
     */
    private ObjectBinding<Image> shipSideImageBinding() {
        return new ObjectBinding<Image>(){
            {
                super.bind(currentShipTypeIndex);
            }
            @Override
            protected Image computeValue() {
                switch (getCurrentShipType()) {
                    case SNAIKKA:
                        return imageLoader.getImage("images/schnikka_side");
                    case CRAYER:
                        return imageLoader.getImage("images/crayer_side");
                    case COG:
                        return imageLoader.getImage("images/cog_side");
                    case HOLK:
                        return imageLoader.getImage("images/holk_side");
                    default:
                        throw new RuntimeException("Unknown ships type: "+ getCurrentShipType());
                }
            }
        };
    }
    /**
     * Binding for the appropriate front image of the ship.
     * @return  .
     */
    private ObjectBinding<Image> shipFrontImageBinding() {
        return new ObjectBinding<Image>(){
            {
                super.bind(currentShipTypeIndex);
            }
            @Override
            protected Image computeValue() {
                switch (getCurrentShipType()) {
                    case SNAIKKA:
                        return imageLoader.getImage("images/schnikka_front");
                    case CRAYER:
                        return imageLoader.getImage("images/crayer_front");
                    case COG:
                        return imageLoader.getImage("images/cog_front");
                    case HOLK:
                        return imageLoader.getImage("images/holk_front");
                    default:
                        throw new RuntimeException("Unknown ships type: "+ getCurrentShipType());
                }
            }
        };
    }

    /**
     * Retrieve the currently selected ship type.
     * @return  current ship type
     */
    protected final EShipType getCurrentShipType() {
        return shipTypes[currentShipTypeIndex.get()];
    }


    public IntegerProperty currentShipTypeIndexProperty() {
        return currentShipTypeIndex;
    }

    /**
     * Indication if there are ships and the dialog can be rendered.
     * Default value is true. Subclasses should overwrite this method.
      * @return true if there are ships in port or this fact is not relevant.
     */
    protected boolean hasShips() {
        return true;
    }

    @Override
    protected void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }
    public ICity getCity() {
        return city.getCity();
    }
    public IPlayer getPlayer() {
        return city.getPlayer();
    }
    public IShip getActiveShip() {
        return city.getActiveShip();
    }
    public Font getFont() {
        return font;
    }
    protected class PriceBinding extends IntegerBinding {
        private final IWare ware;
        private final IntegerProperty available;
        private final BuyAmountBinding buyAmount;

        public PriceBinding(IWare ware, BuyAmountBinding buyAmount) {
            this.ware = ware;
            available = getCity().getWare(ware).amountProperty();
            this.buyAmount = buyAmount;
            super.bind(buyAmount, available);
        }

        @Override
        protected int computeValue() {
            if (buyAmount.get() > 0) {
                return ware.buyPrice(available, buyAmount);
            } else {
                return 0;
            }
        }
    }

    /**
     * Binding of the amount that has to be bought as it is not stored in a warehouse.
     */
    protected class BuyAmountBinding extends IntegerBinding {
        private final IntegerProperty stored;
        private final IntegerBinding needed;
        public BuyAmountBinding(IWare ware, IntegerBinding requiredAmount) {
            final ITradingOffice office = getPlayer().findTradingOffice(getCity());
            stored = office.getWare(ware).amountProperty();
            needed = requiredAmount;
            super.bind(stored, requiredAmount);
        }

        @Override
        protected int computeValue() {
            return Math.max(0, needed.get() - stored.get());
        }
    }

    private class ReplaceFirstPage implements EventHandler<MouseEvent> {
        private final List<Node> firstPageContent;
        public ReplaceFirstPage(List<Node> firstPageContent) {
            this.firstPageContent = firstPageContent;
        }

        @Override
        public void handle(MouseEvent mouseEvent) {
            replaceContent(firstPageContent);
        }
    }
}
