package ch.sahits.game.graphic.display.dialog.shipyard;

import ch.sahits.game.event.NoticeBoardClose;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.Pane;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.Arrays;
import java.util.List;

/**
 * Listing all the costruction ships and provide feature to cancel the order.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 17, 2013
 */
@Component("shipyardConstructionListDialog")
@Scope(value = "prototype")
@ClassCategory({EClassCategory.JAVAFX, EClassCategory.SINGLETON_BEAN})
public class ShipyardConstructionListDialogV2 extends CloseButtonDialog {
    /** Reference to the city view model */
    private final ICityPlayerProxyJFX city;
    private final Font font;
    @ObjectPropertyType(IShipDueDate.class)
    private final ObjectProperty<IShipDueDate> selectedShip;
    @Autowired
    private Date date;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;

    private GridPane grid;

    public ShipyardConstructionListDialogV2(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.font = font;
        this.city = city;
        selectedShip = new SimpleObjectProperty<>(this, "selectedShip", null);
    }
    @PostConstruct
    private void initializeModelAndDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.title", new Object[]{}, locale.getCurrentLocal()));
        final IShipyard state = city.getCity().getCityState().getShipyardState();
        if (state.getShipBuildingList().isEmpty()) {
           Text text = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.noShipsBuilding", new Object[]{}, locale.getCurrentLocal()));
            text.setFont(font);
            text.setLayoutX(50);
            text.setLayoutY(100);
            getContent().addAll(text);
        } else {
            grid = new GridPane();
            grid.setLayoutX(2*FRAME_BORDER);
            grid.setLayoutY(100);
            Text headerCol1 = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.dueDate", new Object[]{}, locale.getCurrentLocal()));
            headerCol1.setFont(font);
            grid.add(headerCol1, 0, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(100));
            Text headerCol2 = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.owner", new Object[]{}, locale.getCurrentLocal()));
            headerCol2.setFont(font);
            grid.add(headerCol2, 1, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(250));
            Text headerCol3 = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.shipType", new Object[]{}, locale.getCurrentLocal()));
            headerCol3.setFont(font);
            grid.add(headerCol3, 2, 0);
            grid.getColumnConstraints().add(new ColumnConstraints(100));

            for (int row = 1; row <= state.getShipBuildingList().size(); row++) {
                final int index = row - 1;
                IShipDueDate dueDate = state.getShipBuildingList().get(index);
                DateTime time = dueDate.getDueDate();
                final IShip ship = dueDate.getShip();
                final IPlayer owner = (IPlayer) ship.getOwner();
                boolean isOwnedByPlayer = owner.equals(city.getPlayer());
                Text text1 = new Text(date.todisplayString(time.getDayOfMonth(), time.getMonthOfYear(), time.getYear()));
                text1.setFont(font);
                text1.setLayoutY(15);
                Pane col1 = new Pane();
                col1.getChildren().add(text1);
                grid.add(col1, 0, row);
                Text text2 = new Text(owner.getName()+" "+ owner.getLastName());
                text2.setFont(font);
                text2.setLayoutY(15);
                Pane col2 = new Pane();
                col2.getChildren().add(text2);
                grid.add(col2, 1, row);
                Text text3 = new Text(ship.getShipType().name());
                text3.setFont(font);
                text3.setLayoutY(15);
                Pane col3 = new Pane();
                col3.getChildren().add(text3);
                grid.add(col3, 2, row);

                SelectLineEventHandler selectionLineEventHandler = new SelectLineEventHandler(dueDate, col1, col2, col3);

                if (isOwnedByPlayer) {
                    text1.setOnMouseReleased(selectionLineEventHandler);
                    text2.setOnMouseReleased(selectionLineEventHandler);
                    text3.setOnMouseReleased(selectionLineEventHandler);
                }
            } // end for

            final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.ShipyardConstructionListDialogV2.cancel", new Object[]{}, locale.getCurrentLocal()), font);
            action.setOnAction(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                     state.cancelShipBuildingOrder(selectedShip.get().getDueDate());
                    executeOnCloseButtonClicked();
                }
            });
            final int actionButtonX = (WIDTH - 124) / 2;
            action.setLayoutX(actionButtonX);
            action.setLayoutY(CLOSE_BTN_Y_POS - 24);
            BooleanBinding actionEnabled = new BooleanBinding() {
                {
                    super.bind(selectedShip);
                }
                @Override
                protected boolean computeValue() {
                    return selectedShip.get() != null;
                }
            };
            action.setDisable(true);
            actionEnabled.addListener(new ChangeListener<Boolean>() {
                @Override
                public void changed(ObservableValue<? extends Boolean> observableValue,
                                    Boolean oldValue, Boolean newValue) {
                    action.setDisable(!newValue);
                }
            });

            getContent().addAll(grid, action);
        }
    }

    private class SelectLineEventHandler implements EventHandler<MouseEvent> {
        private final IShipDueDate ship;
        private final List<Pane> row;

        private SelectLineEventHandler(IShipDueDate ship, Pane...row) {
            this.ship = ship;
            this.row = Arrays.asList(row);
        }

        @Override
        public void handle(MouseEvent mouseEvent) {
            selectedShip.set(ship);
            for (Node node : grid.getChildren()) {
                if (row.contains(node)) {
                    node.setStyle("-fx-background-color: grey");
                } else {
                    node.setStyle("-fx-background-color: transparent");
                }
            }
        }
    }
    @Override
    protected void executeOnCloseButtonClicked() {
        clientEventBus.post(new NoticeBoardClose());
        super.executeOnCloseButtonClicked();
    }

}
