package ch.sahits.game.graphic.display.dialog.tavern;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.dialog.util.DialogUtil;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.people.ISailorState;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.beans.value.WeakChangeListener;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.VBox;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;

/**
 * Dialog for hiring sailors.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernSailorDialog extends CloseButtonDialog {

    private final ICityPlayerProxyJFX city;
    private final Font font;
    private final ISailorState sailorState;
    @ObjectPropertyType(IShip.class)
    private final ObjectProperty<IShip> currentShip;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;

    @Autowired
    private MessageSource messageSource;

    private ChangeListener<Number> sailorNumberListener;

    public TavernSailorDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
        sailorState = city.getCity().getCityState().getTavernState().getSailors();
        this.currentShip = new SimpleObjectProperty<>(this, "currentShip", city.getActiveShip());
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.title", new Object[]{}, locale.getCurrentLocal()));
        VBox topText = new VBox();
        topText.setLayoutX(2 * FRAME_BORDER);
        topText.setLayoutY(100);

        Text firstLine = new Text();
        firstLine.setFont(font);
        StringBinding firstLineBinding = new StringBinding() {
            {
                super.bind(sailorState.numberOfSailorsProperty());
            }
            @Override
            protected String computeValue() {
                if (sailorState.getNumberOfSailors() > 1) {
                    return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.dialog1", new Object[]{ sailorState.getNumberOfSailors()}, locale.getCurrentLocal());
                } else if (sailorState.getNumberOfSailors() == 1) {
                    return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.dialog2", new Object[]{}, locale.getCurrentLocal());
                } else {
                    return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.dialog3", new Object[]{}, locale.getCurrentLocal());
                }
            }
        };
        firstLine.textProperty().bind(firstLineBinding);
        Text secondLine = new Text();
        secondLine.setFont(font);
        secondLine.setText(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.dialog4", new Object[]{}, locale.getCurrentLocal()));
        topText.getChildren().addAll(firstLine, secondLine);
        getContent().add(topText);

        final IntegerProperty sailorsAdditionalOnShip = new SimpleIntegerProperty(0);
        final IntegerProperty sailorsInTavern = new SimpleIntegerProperty(sailorState.getNumberOfSailors());

        currentShip.addListener((observableValue, oldValue, newValue) -> {
            sailorsAdditionalOnShip.set(newValue.getNumberOfSailors());
            sailorsInTavern.set(sailorState.getNumberOfSailors()); // reset
        });
        sailorNumberListener = (observableValue, oldValue, newValue) -> {
                int inTavern = newValue.intValue() - sailorsAdditionalOnShip.get();
                if (inTavern < 0) {
                    sailorsAdditionalOnShip.set(Math.max(sailorsAdditionalOnShip.get() + inTavern, 0));
                    inTavern = 0;
                }
                sailorsInTavern.set(inTavern);
        };
        sailorState.numberOfSailorsProperty().addListener(new WeakChangeListener<>(sailorNumberListener));


        GridPane sailorsPane = createSailorMoveControls(sailorsAdditionalOnShip, sailorsInTavern);
        getContent().add(sailorsPane);

        Text selectedShip = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.dialog5", new Object[]{}, locale.getCurrentLocal()));
        selectedShip.setFont(font);
        selectedShip.setLayoutX(2 * FRAME_BORDER);
        selectedShip.setLayoutY(250);
        getContent().add(selectedShip);

        if (currentShip.get() == null) {
            Text noShip = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernCaptainDialog.noShip", new Object[]{}, locale.getCurrentLocal()));
            noShip.setFont(font);
            noShip.setLayoutX(2*FRAME_BORDER);
            noShip.setLayoutY(300);
            getContent().add(noShip);
        } else {
            GridPane shipSelectionPane = dialogUtil.createShipSelection3Lines(city, currentShip, font);
            shipSelectionPane.setLayoutX(2*FRAME_BORDER);
            shipSelectionPane.setLayoutY(300);
            getContent().add(shipSelectionPane);
        }

        final int actionButtonX = (WIDTH - 124) / 2;
        final OpenPatricianLargeWaxButton action = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.hire", new Object[]{}, locale.getCurrentLocal()), font);
        action.setOnAction(mouseEvent -> {
            currentShip.get().setNumberOfSailors(sailorsAdditionalOnShip.get() + currentShip.get().getNumberOfSailors());
            sailorsAdditionalOnShip.setValue(0);
            sailorState.setNumberOfSailors(sailorsInTavern.get());
            executeOnCloseButtonClicked();
        });
        action.setLayoutX(actionButtonX);
        action.setLayoutY(CLOSE_BTN_Y_POS - 24);
        BooleanBinding enableAction = new BooleanBinding() {
            {
                super.bind(currentShip,sailorsAdditionalOnShip);
            }
            @Override
            protected boolean computeValue() {
                return currentShip.get() != null
                        && sailorsAdditionalOnShip.get() > 0;
            }
        };
        action.setDisable(!enableAction.get());
        enableAction.addListener((observableValue, oldValue, newValue) -> {
             action.setDisable(!newValue);
        });
        getContent().add(action);


    }


    /**
     * Create the grid pane for moving the sailors between the tavern and the ship.
     * The sailors are only updated, when they are hired.
     * @param sailorsOnShip property indicating the current sailors assigned to the ship (might
     *                      differ from the number actually on the ship)
     * @param sailorsInTavern property indicating the current sailors in the tavern.
     * @return Grid pane.
     */
    private GridPane createSailorMoveControls(final IntegerProperty sailorsOnShip, final IntegerProperty sailorsInTavern) {
        GridPane sailorsPane = new GridPane();
        sailorsPane.setLayoutX(100);
        sailorsPane.setLayoutY(150);
        RowConstraints rowConstraint = new RowConstraints(24);
        sailorsPane.getRowConstraints().addAll(rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        sailorsPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint);
        Text tavernTxt = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernSailorDialog.tavern", new Object[]{}, locale.getCurrentLocal()));
        tavernTxt.setFont(font);
        sailorsPane.add(tavernTxt, 0, 0);
        GridPane.setHalignment(tavernTxt, HPos.CENTER);
        Text shipTxt = new Text(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.ship", new Object[]{}, locale.getCurrentLocal()));
        shipTxt.setFont(font);
        sailorsPane.add(shipTxt, 3, 0);
        GridPane.setHalignment(shipTxt, HPos.CENTER);

        Text nbInTavern = new Text();
        nbInTavern.setFont(font);
        nbInTavern.textProperty().bind(sailorsInTavern.asString());
        sailorsPane.add(nbInTavern, 0, 1);
        GridPane.setHalignment(nbInTavern, HPos.CENTER);
        final OpenPatricianSmallWaxButton toTavernBtn = new OpenPatricianSmallWaxButton("<");
        toTavernBtn.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                if(sailorsOnShip.get()>0) {
                    sailorsOnShip.set(sailorsOnShip.get() - 1);
                    sailorsInTavern.set(sailorsInTavern.get() + 1);
                }
            }
        });
        if (sailorsOnShip.get() < 1 || currentShip.get() == null) {
            toTavernBtn.setDisable(true);
        }
        sailorsPane.add(toTavernBtn, 1, 1);
        GridPane.setHalignment(toTavernBtn, HPos.CENTER);
        final OpenPatricianSmallWaxButton toShipBtn = new OpenPatricianSmallWaxButton(">");
        toShipBtn.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                if(sailorsInTavern.get()>0) {
                    sailorsOnShip.set(sailorsOnShip.get() + 1);
                    sailorsInTavern.set(sailorsInTavern.get() - 1);
                }
            }
        });
        sailorsOnShip.addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue,
                                Number oldValue, Number newValue) {
                if (newValue.intValue() <= 0) {
                    toTavernBtn.setDisable(true);
                } else {
                    toTavernBtn.setDisable(false);
                }
                if (sailorsInTavern.get() <= 0 || newValue.intValue() + currentShip.get().getNumberOfSailors() >= currentShip.get().getMaxNumberOfSailors()) {
                    toShipBtn.setDisable(true);
                } else {
                    toShipBtn.setDisable(false);
                }
            }
        });
        sailorsInTavern.addListener(new ChangeListener<Number>() {
            @Override
            public void changed(ObservableValue<? extends Number> observableValue,
                                Number oldValue, Number newValue) {
                if (newValue.intValue() <= 0 || sailorsOnShip.get() + currentShip.get().getNumberOfSailors() >= currentShip.get().getMaxNumberOfSailors()) {
                    toShipBtn.setDisable(true);
                } else {
                    toShipBtn.setDisable(false);
                }

            }
        });
        if (sailorsInTavern.get() < 1 || currentShip.get() == null || currentShip.get().getNumberOfSailors() == currentShip.get().getMaxNumberOfSailors()) {
            toShipBtn.setDisable(true);
        }
        sailorsPane.add(toShipBtn, 2, 1);
        GridPane.setHalignment(toShipBtn, HPos.CENTER);

        Text nbInShip = new Text();
        nbInShip.setFont(font);
        nbInShip.textProperty().bind(sailorsOnShip.asString());
        sailorsPane.add(nbInShip, 3, 1);
        GridPane.setHalignment(nbInShip, HPos.CENTER);
        return sailorsPane;
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }

}
