package ch.sahits.game.graphic.display.dialog.tavern;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.dialog.util.TavernDialogUtil;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianLargeWaxButton;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.people.ITrader;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.WeakChangeListener;
import javafx.scene.text.Font;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;

/**
 * Dialog for a trader in the tavern.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 25, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernTraderDialog extends CloseButtonDialog {
    private final ICityPlayerProxyJFX city;
    private final Font font;
    private final ITrader trader;
    @Autowired
    private TavernDialogUtil tavernDialogUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;

    private final TavernState tavernState;
    private ChangeListener<Boolean> leaving;

    public TavernTraderDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
        tavernState = city.getCity().getCityState().getTavernState();
        trader = tavernState.getTrader();
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernTraderDialog.title", new Object[]{}, locale.getCurrentLocal()));
        final IWare ware = trader.getWare();
        String loadedText;
        if (ware.isBarrelSizedWare()) {
            loadedText = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernTraderDialog.dialogBarrel", new Object[]{city.getPlayer().getLastName(), translator.getLocalDisplayName((EWare) ware)}, locale.getCurrentLocal());
        } else {
            loadedText = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernTraderDialog.dialogBale", new Object[]{city.getPlayer().getLastName(), translator.getLocalDisplayName((EWare) ware)}, locale.getCurrentLocal());
        }


        HashMap<String, Object> parameters = new HashMap<>();
        parameters.put("amount", trader.getAmount());
        parameters.put("price", trader.getAvgPricePerItem());
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        final OpenPatricianLargeWaxButton acceptButton = new OpenPatricianLargeWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernBaseSideRoomPersonDialog.accept", new Object[]{}, locale.getCurrentLocal()), font);
        int BUTTON_X = (WIDTH - 124) / 2;
        acceptButton.setLayoutX(BUTTON_X);
        int LOWER_BUTTON_Y = CLOSE_BTN_Y_POS - 24;
        acceptButton.setLayoutY(LOWER_BUTTON_Y);
        acceptButton.setOnAction(mouseEvent -> {
            IPlayer player = city.getPlayer();
            int price = trader.getAmount() * trader.getAvgPricePerItem();
            ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
            if (office != null) {
                office.move(trader.getWare(), trader.getAmount());
            } else {
                IShip ship = city.getActiveShip();
                if (ship != null) {
                    if (ship.getCapacity()>=trader.getAmount()) {
                        ship.load(trader.getWare(), trader.getAmount(), trader.getAvgPricePerItem());
                    } else {
                        acceptButton.setDisable(true);
                        return;
                    }
                }  else {
                    acceptButton.setDisable(true);
                    return;
                }
            }
            player.getCompany().updateCash(-price);
            trader.leave();
            executeOnCloseButtonClicked();
        });
        BooleanBinding disable = disableAcceptBinding();
        acceptButton.setDisable(disable.get());
        disable.addListener((observableValue, oldValue, newValue) -> {
            acceptButton.setDisable(newValue);
        });
        leaving = (observable, oldValue, newValue) -> {
            if (!newValue) { // the person became unavailable
                executeOnCloseButtonClicked();
            }
        };
        trader.isPresentProperty().addListener(new WeakChangeListener<>(leaving));

        getContent().addAll(dt, acceptButton);
    }

    private BooleanBinding disableAcceptBinding() {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
                IShip ship = city.getActiveShip();
                int barrelAmountFactor = 1;
                if (!trader.getWare().isBarrelSizedWare()) {
                    barrelAmountFactor = 10;
                }
                return office == null && ship == null && ship.getCapacity() < trader.getAmount() * barrelAmountFactor;
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        tavernDialogUtil.stopTalkingToPerson(tavernState, city.getPlayer());
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }
}
