package ch.sahits.game.graphic.display.dialog.tavern;

import ch.sahits.game.event.EViewChangeEvent;
import ch.sahits.game.event.NoticeBoardUpdate;
import ch.sahits.game.graphic.display.ClientViewState;
import ch.sahits.game.graphic.display.dialog.CloseButtonDialog;
import ch.sahits.game.graphic.display.dialog.util.DialogUtil;
import ch.sahits.game.graphic.display.model.ViewChangeCityPlayerProxyJFX;
import ch.sahits.game.javafx.control.DecoratedText;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.model.ControlTableCell;
import ch.sahits.game.javafx.model.StaticTextTableCell;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.model.TableHeader;
import ch.sahits.game.javafx.model.TableRow;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.ObjectPropertyType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.people.IWeaponsDealer;
import ch.sahits.game.openpatrician.model.product.ETransferAmount;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.util.ModelTranslations;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.util.TransferUtil;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import com.google.common.eventbus.AsyncEventBus;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.IntegerBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.IntegerProperty;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleIntegerProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import java.util.HashMap;

/**
 * Dialog for dealing weapons in the tavern.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 21, 2013
 */
@Prototype
@ClassCategory({EClassCategory.DIALOG, EClassCategory.PROTOTYPE_BEAN, EClassCategory.UNRELEVANT_FOR_DESERIALISATION})
public class TavernWeaponsDealerDialog extends CloseButtonDialog {
    @ObjectPropertyType(ETransferAmount.class)
    private ObjectProperty<ETransferAmount> movableAmount = new SimpleObjectProperty<>(ETransferAmount.ONE);
    private final ICityPlayerProxyJFX city;
    private final Font font;
    @ListType(Object.class) // IShip and ITradingOffice create better generic type
    private ObservableList<Object> storage = FXCollections.observableArrayList();
    private IntegerProperty currentStorageIndex = new SimpleIntegerProperty(-1);
    private StringBinding storageDestinationBinding;
    @Autowired
    private DialogUtil dialogUtil;
    @Autowired
    private TransferUtil transferUtil;
    @Autowired
    private ClientViewState viewState;
    @Autowired
    @Qualifier("clientEventBus")
    private AsyncEventBus clientEventBus;
    @Autowired
    private Locale locale;
    @Autowired
    private ModelTranslations translator;
    @Autowired
    private MessageSource messageSource;

    private EWeapon[] weapons = {EWeapon.SWORD, EWeapon.BOW, EWeapon.CROSSBOW, EWeapon.MUSKET};

    private final IWeaponsDealer weaponsDealer;

    public TavernWeaponsDealerDialog(Font font, ICityPlayerProxyJFX city) {
        super(font);
        this.city = city;
        this.font = font;
        weaponsDealer = city.getCity().getCityState().getTavernState().getWeaponsDealer();
        initializeStorage();
        initStorageDestionationBinding();

    }

    private void initializeStorage() {
        IShip activeShip = city.getActiveShip();
        if (activeShip != null) {
            storage.add(activeShip);
        }
        for (IShip ship : city.getPlayersShips()) {
            if (!ship.equals(activeShip)) {
                storage.add(ship);
            }
        }
        ITradingOffice office = city.getPlayer().findTradingOffice(city.getCity());
        if (office != null) {
            storage.add(office);
        }
        if (!storage.isEmpty()) {
            currentStorageIndex.set(0);
        }
        // todo: andi 21/12/13: add listener to the players ship to update as they leave/arrive
    }
    @PostConstruct
    private void initializeDialog() {
        setTitle(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernWeaponsDealerDialog.title", new Object[]{}, locale.getCurrentLocal()));

        String loadedText = messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernWeaponsDealerDialog.dialog1", new Object[0], locale.getCurrentLocal());


        HashMap<String, Object> parameters = new HashMap<>();
        DecoratedText dt = decoratedTextFactory.createDecoratedText(loadedText, parameters);


        getContent().addAll(dt);

        GridPane catalogue = createShipCatalogue();
        getContent().add(catalogue);

        GridPane weapnsTable = createWeaponsTable();
        getContent().add(weapnsTable);

        OpenPatricianSmallWaxButton btn1 = new OpenPatricianSmallWaxButton("1", font);
        btn1.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent arg0) {
                movableAmount.setValue(ETransferAmount.ONE);
            }
        });
        OpenPatricianSmallWaxButton btn5 = new OpenPatricianSmallWaxButton("5", font);
        btn5.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent arg0) {
                movableAmount.setValue(ETransferAmount.FIVE);
            }
        });
        OpenPatricianSmallWaxButton btnMax = new OpenPatricianSmallWaxButton(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.max", new Object[]{}, locale.getCurrentLocal()), font);
        btnMax.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent arg0) {
                movableAmount.setValue(ETransferAmount.MAX);
            }
        });
        addButtomControlButtons(btn1, btn5,btnMax);
    }
    private GridPane createWeaponsTable() {

        Table model = new Table();
        TableHeader header = new TableHeader(5);
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernWeaponsDealerDialog.weaponType", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernWeaponsDealerDialog.dealer", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.buy", new Object[]{}, locale.getCurrentLocal())));
        header.add(new StaticTextTableCell(messageSource.getMessage("ch.sahits.game.graphic.display.dialog.BaseTradeDialog.sell", new Object[]{}, locale.getCurrentLocal())));
        Text storageHeader = new Text();
        storageHeader.setFont(font);
        storageHeader.textProperty().bind(storageDestinationBinding);
        header.add(new ControlTableCell(storageHeader));
        header.setAligenment(0, HPos.CENTER);
        header.setAligenment(1, HPos.CENTER);
        header.setAligenment(2, HPos.CENTER);
        header.setAligenment(3, HPos.CENTER);
        header.setAligenment(4, HPos.CENTER);
        model.setHeader(header);
        model.setAligenment(0, HPos.RIGHT);
        model.setAligenment(1, HPos.CENTER);
        model.setAligenment(2, HPos.CENTER);
        model.setAligenment(3, HPos.CENTER);
        model.setAligenment(4, HPos.CENTER);
        model.setColumnWidth(100, 70, 70, 70, 100);

        for (final EWeapon weapon : weapons) {
            final IntegerBinding storedAmount = getAmountStoredWeapons(weapon); // amount owned by the player
            final IntegerBinding amountToSell = getAmountBinding(storedAmount);
            final IntegerProperty dealerOwnedAmount = weaponsDealer.amountAvailableProperty(weapon);

            TableRow row = new TableRow();
            row.add(new StaticTextTableCell(translator.getLocalDisplayName(weapon)));
            Text dealerAmount = new Text();
            dealerAmount.setFont(font);
            dealerAmount.textProperty().bind(dealerOwnedAmount.asString());
            row.add(new ControlTableCell(dealerAmount));
            OpenPatricianSmallWaxButton buyButton = new OpenPatricianSmallWaxButton("", font);
            final IntegerBinding buyPriceBinding = weaponsDealer.buyPriceBinding(movableAmount, weapon);
            buyButton.textProperty().bind(buyPriceBinding.asString());
            buyButton.setOnAction(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    if (dealerOwnedAmount.get() > 0 ) {
                        int amountToBuy = transferUtil.calculateAvailableAmount(movableAmount, dealerOwnedAmount.get());
                        int avgPrice = buyPriceBinding.get();
                        city.getPlayer().getCompany().updateCash(-amountToBuy * avgPrice);
                        if (storage.get(currentStorageIndex.get()) instanceof ITradingOffice) {
                            ITradingOffice office = (ITradingOffice) storage.get(currentStorageIndex.get());
                            office.getWeaponStorage().update(weapon, amountToBuy);
                        } else {
                            IShip ship = (IShip) storage.get(currentStorageIndex.get());
                            ship.move(weapon, amountToBuy);
                        }
                        weaponsDealer.move(weapon, -amountToBuy);
                        storedAmount.invalidate();
                    }
                }
            });
            row.add(new ControlTableCell(buyButton));


            OpenPatricianSmallWaxButton sellButton = new OpenPatricianSmallWaxButton("", font);
            final IntegerBinding sellPriceBinding = weaponsDealer.sellPriceBinding(weapon, amountToSell);
            sellButton.textProperty().bind(sellPriceBinding.asString());
            sellButton.setOnAction(new EventHandler<MouseEvent>() {
                @Override
                public void handle(MouseEvent mouseEvent) {
                    int amount = amountToSell.get();
                    int avgPrice = sellPriceBinding.get();
                    city.getPlayer().getCompany().updateCash(amount * avgPrice);
                    if (storage.get(currentStorageIndex.get()) instanceof ITradingOffice) {
                        ITradingOffice office = (ITradingOffice) storage.get(currentStorageIndex.get());
                        office.getWeaponStorage().update(weapon, -amount);
                    } else {
                        IShip ship = (IShip) storage.get(currentStorageIndex.get());
                        ship.move(weapon, -amount);
                    }
                    weaponsDealer.move(weapon, amount);
                    storedAmount.invalidate();
                }
            });
            row.add(new ControlTableCell(sellButton));

            Text amountStored = new Text();
            amountStored.setFont(font);
            amountStored.textProperty().bind(storedAmount.asString());
            row.add(new ControlTableCell(amountStored));

            model.add(row);
        }

        GridPane pane = dialogUtil.createGridPaneFromModel(model, font);
        pane.setLayoutX(2*FRAME_BORDER);
        pane.setLayoutY(350);
        return pane;
    }


    private GridPane createShipCatalogue() {
        GridPane shipCatalogue = new GridPane();
        shipCatalogue.setLayoutX(2*FRAME_BORDER);
        shipCatalogue.setLayoutY(300);
        shipCatalogue.getColumnConstraints().addAll(new ColumnConstraints(64), new ColumnConstraints(WIDTH - 4*FRAME_BORDER - 2* 64), new ColumnConstraints(64));
        BooleanBinding enablePrevNext = new BooleanBinding() {
            {
                super.bind(storage);
            }
            @Override
            protected boolean computeValue() {
                return storage.size() > 1;
            }
        };

        final OpenPatricianSmallWaxButton prevShip = new OpenPatricianSmallWaxButton("<", font);
        prevShip.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                if (storage.size() > 1) {
                    if (currentStorageIndex.get() == 0) {
                        currentStorageIndex.set(storage.size() -1); // last one;
                    } else {
                        currentStorageIndex.set(currentStorageIndex.get() - 1);
                    }
                }
            }
        });
        prevShip.setDisable(!enablePrevNext.get());
        shipCatalogue.add(prevShip, 0, 0);
        final OpenPatricianSmallWaxButton nextShip = new OpenPatricianSmallWaxButton(">", font);
        nextShip.setOnAction(new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                if (storage.size() > 1) {
                    if (currentStorageIndex.get() == storage.size() -1) {
                        currentStorageIndex.set(0);
                    } else {
                        currentStorageIndex.set(currentStorageIndex.get() + 1);
                    }
                }
            }
        });
        nextShip.setDisable(!enablePrevNext.get());
        shipCatalogue.add(nextShip, 2, 0);
        enablePrevNext.addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observableValue,
                                Boolean oldValue, Boolean newValue) {
                nextShip.setDisable(!newValue);
                prevShip.setDisable(!newValue);
            }
        });
        final Text ship = new Text();
        ship.setFont(font);
        ship.textProperty().bind(storageDestinationBinding);
        shipCatalogue.add(ship, 1, 0);
        GridPane.setHalignment(ship, HPos.CENTER);
        return shipCatalogue;
    }

    private void initStorageDestionationBinding() {
        if (storageDestinationBinding == null) {
            storageDestinationBinding = new StringBinding() {
                {
                    super.bind(currentStorageIndex);
                }

                @Override
                protected String computeValue() {
                    final Object holding = storage.get(currentStorageIndex.get());
                    if (holding instanceof IShip) {
                        return ((IShip) holding).getName();
                    } else if (holding instanceof ITradingOffice) {
                        return messageSource.getMessage("ch.sahits.game.graphic.display.dialog.TavernWeaponsDealerDialog.tradingOffice", new Object[]{}, locale.getCurrentLocal());
                    } else {
                        throw new RuntimeException("This type should not be here.");
                    }
                }
            };
        }
    }

    /**
     * Retrieve the binding for the amount of the weapon currently stored.
     * @param weapon type of the weapon
     * @return integer binding of the stored amount of the weapon
     */
    private IntegerBinding getAmountStoredWeapons(final EWeapon weapon) {
        return new IntegerBinding() {
            {
                super.bind(storage, currentStorageIndex);
            }
            @Override
            protected int computeValue() {
                if (storage.get(currentStorageIndex.get()) instanceof ITradingOffice) {
                    ITradingOffice office = (ITradingOffice) storage.get(currentStorageIndex.get());
                    return office.getWeaponStorage().getWeapon(weapon);
                } else {
                    IShip ship = (IShip) storage.get(currentStorageIndex.get());
                    return ship.getWeaponAmount(weapon);
                }
            }
        };
    }
    /**
     * Bind the amount that can be transferred on the maximal available amount
     * as well as the selected transfer amount. This reflects the amount the
     * dealer owns.
     * @param maxAmount binding of the amount the dealer owns
     * @return integer binding
     */
    private IntegerBinding getAmountBinding(final IntegerBinding maxAmount) {
        return new IntegerBinding() {
            {
                super.bind(movableAmount, maxAmount);
            }

            @Override
            protected int computeValue() {
                return transferUtil.calculateAvailableAmount(movableAmount, maxAmount.get());
            }
        };
    }
    /**
     * Update the notice board and close the dialog.
     */
    @Override
    protected void executeOnCloseButtonClicked() {
        ViewChangeCityPlayerProxyJFX proxy = new ViewChangeCityPlayerProxyJFX(viewState.getCurrentCityProxy().get(), EViewChangeEvent.NOTICE_TAVERN);
        clientEventBus.post(new NoticeBoardUpdate(proxy));
        super.executeOnCloseButtonClicked();
    }


}
