package ch.sahits.game.graphic.display.dialog.util;

import ch.sahits.game.javafx.control.BarrelAmount;
import ch.sahits.game.javafx.control.OpenPatricianSmallWaxButton;
import ch.sahits.game.javafx.model.Table;
import ch.sahits.game.javafx.util.JavaFXUtils;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.model.ship.IShip;
import javafx.beans.Observable;
import javafx.beans.binding.BooleanBinding;
import javafx.beans.binding.StringBinding;
import javafx.beans.property.ObjectProperty;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.RowConstraints;
import javafx.scene.text.Font;
import javafx.scene.text.Text;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 *         Created on Dec 20, 2013
 */
@Component
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class DialogUtil {
    @Autowired
    private JavaFXUtils fxUtils;
    /**
     * Boolean binding for the navigation enabling of the ships cataloque.
     * @param city proxy
     * @return binding.
     */
    public BooleanBinding enableShipCatalogue(final ICityPlayerProxyJFX city) {
        return new BooleanBinding() {
            {
                super.bind(city.getPlayersShips());
            }
            @Override
            protected boolean computeValue() {
                return city.getPlayersShips().size() > 1;
            }
        };
    }

    /**
     * Navigate to the next ship in the catalogue
     * @param city proxy
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */
    public EventHandler<MouseEvent> createNextAction(final ICityPlayerProxyJFX city,
                                                        final ObjectProperty<IShip> currentShip) {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                List<IShip> ships = city.getPlayersShips();
                if (ships.size() > 1) {
                    if (ships.get(ships.size()-1).equals(currentShip.get())) {
                        currentShip.set(ships.get(0)); // first one;
                    } else {
                        for (int i=0;i<ships.size()-1; i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip nextShip = ships.get(i + 1);
                                currentShip.set(nextShip);
                                break;
                            }
                        }
                    }
                }
            }
        };
    }
    /**
     * Navigate to the previous ship in the catalogue
     * @param city proxy
     * @param currentShip current ship
     * @return Event handler for the navigation.
     */

    public EventHandler<MouseEvent> createPreviousAction(final ICityPlayerProxyJFX city,
                                                         final ObjectProperty<IShip> currentShip) {
        return new EventHandler<MouseEvent>() {
            @Override
            public void handle(MouseEvent mouseEvent) {
                List<IShip> ships = city.getPlayersShips();
                if (ships.size() > 1) {
                    if (ships.get(0).equals(currentShip.get())) {
                        currentShip.set(ships.get(ships.size()-1)); // last one;
                    } else {
                        for (int i=1;i<ships.size(); i++) {
                            if (ships.get(i).equals(currentShip.get())) {
                                final IShip prevShip = ships.get(i - 1);
                                currentShip.set(prevShip);
                                break;
                            }
                        }
                    }
                }
            }
        };
    }

    /**
     * Create a grid pane based on a model.
     * @param model base model
     * @param font to be used
     * @return grid pane
     */
    public GridPane createGridPaneFromModel(Table model, Font font) {
        return fxUtils.createGridPaneFromModel(model, font);
    }
    /**
     * Create the grid pane for the ship selection on three lines.
     * @param city proxy for the player city
     * @param currentShip non empty current ship
     * @param font to be used
     * @return grid pane
     */
    public GridPane createShipSelection3Lines(final ICityPlayerProxyJFX city, final ObjectProperty<IShip> currentShip, Font font) {
        GridPane shipSelectionPane = new GridPane();
        RowConstraints rowConstraint = new RowConstraints(24);
        shipSelectionPane.getRowConstraints().addAll(rowConstraint, rowConstraint, rowConstraint);
        ColumnConstraints colConstraint = new ColumnConstraints(64);
        shipSelectionPane.getColumnConstraints().addAll(colConstraint, colConstraint, colConstraint, colConstraint, colConstraint);
        Text shipName = new Text();
        StringBinding shipNameBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                final IShip ship = currentShip.get();
                return ship.getShipType().name()+" "+ ship.getName();
            }
        };
        shipName.setFont(font);
        shipName.textProperty().bind(shipNameBinding);
        shipSelectionPane.add(shipName,1, 0, 3, 1);
        GridPane.setHalignment(shipName, HPos.CENTER);

        BooleanBinding enablePrevNext = enableShipCatalogue(city);

        final OpenPatricianSmallWaxButton prevShip = new OpenPatricianSmallWaxButton("<");
        prevShip.setOnAction(createPreviousAction(city, currentShip));
        prevShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(prevShip, 0, 1);
        final OpenPatricianSmallWaxButton nextShip = new OpenPatricianSmallWaxButton(">");
        nextShip.setOnAction(createNextAction(city, currentShip));
        nextShip.setDisable(!enablePrevNext.get());
        shipSelectionPane.add(nextShip, 4, 1);
        enablePrevNext.addListener(new ChangeListener<Boolean>() {
            @Override
            public void changed(ObservableValue<? extends Boolean> observableValue,
                                Boolean oldValue, Boolean newValue) {
                nextShip.setDisable(!newValue);
                prevShip.setDisable(!newValue);
            }
        });

        BarrelAmount shipSize = new BarrelAmount(font);
        StringBinding shipSizeBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(currentShip.get().getSize());
            }
        };
        shipSize.amountProperty().bind(shipSizeBinding);
        shipSelectionPane.add(shipSize, 1, 2);
        GridPane.setHalignment(shipSize, HPos.CENTER);

        Text damage = new Text();
        damage.setFont(font);
        StringBinding damageBinding = new StringBinding() {
            {
                super.bind(currentShip);
            }
            @Override
            protected String computeValue() {
                return String.valueOf(100 - currentShip.get().getDamage());
            }
        };
        damage.textProperty().bind(damageBinding);
        shipSelectionPane.add(damage, 2, 2);
        GridPane.setHalignment(damage, HPos.CENTER);

        Text sailorsOnShip2 = new Text();
        sailorsOnShip2.setFont(font);
        final SailorOnShipBinding sailorsOnShipBinding = new SailorOnShipBinding(currentShip);
        currentShip.addListener(new ChangeListener<IShip>() {
            @Override
            public void changed(ObservableValue<? extends IShip> observableValue,
                                IShip oldValue, IShip newValue) {
                sailorsOnShipBinding.delegateUnbind(oldValue.numberOfSailorsProperty());
                sailorsOnShipBinding.delegateBind(newValue.numberOfSailorsProperty());
            }
        });
        sailorsOnShip2.textProperty().bind(sailorsOnShipBinding);
        shipSelectionPane.add(sailorsOnShip2, 3, 2);
        GridPane.setHalignment(sailorsOnShip2, HPos.CENTER);
        return shipSelectionPane;
    }
    private class SailorOnShipBinding extends StringBinding {
        private final ObjectProperty<IShip> currentShip;
        SailorOnShipBinding(ObjectProperty<IShip> currentShip) {
            super.bind(currentShip, currentShip.get().numberOfSailorsProperty());
            this.currentShip = currentShip;
        }
        @Override
        protected String computeValue() {
            final IShip ship = currentShip.get();
            StringBuilder sb = new StringBuilder();
            sb.append(ship.getNumberOfSailors())
                    .append(" (")
                    .append(ship.getMinNumberOfSailors())
                    .append(" - ")
                    .append(ship.getMaxNumberOfSailors())
                    .append(")");
            return sb.toString();
        }
        public void delegateUnbind(Observable...values) {
            unbind(values);
        }
        public void delegateBind(Observable...values) {
            bind(values);
        }
    }
}
