package ch.sahits.game.graphic.display.scene;

import ch.sahits.game.event.EGameStatusChange;
import ch.sahits.game.event.GameStateChange;
import ch.sahits.game.event.data.NewGame;
import ch.sahits.game.graphic.display.SceneChangeable;
import ch.sahits.game.graphic.display.util.GameOptionsService;
import ch.sahits.game.graphic.image.IFontLoader;
import ch.sahits.game.javafx.OpenPatricianScene;
import ch.sahits.game.javafx.control.OpenPatricianSpinner;
import ch.sahits.game.javafx.control.OpenPatricianStoneButton;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.UniquePrototype;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.Difficulty;
import ch.sahits.game.openpatrician.model.EGameSpeed;
import ch.sahits.game.openpatrician.model.EObjective;
import ch.sahits.game.openpatrician.model.GameFactory;
import ch.sahits.game.openpatrician.model.IGame;
import ch.sahits.game.openpatrician.util.l10n.Locale;
import ch.sahits.game.util.UIFactory;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.application.Platform;
import javafx.event.EventHandler;
import javafx.geometry.HPos;
import javafx.geometry.Pos;
import javafx.scene.control.Label;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.StackPane;
import javafx.scene.text.Font;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * Scene for initally selecting a savegame, load it and start it.
 */
@UniquePrototype
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class LoadGameScene extends BaseStartupScene {
	private final Logger logger = LogManager.getLogger(getClass());
	@Autowired
	private UIFactory uiFactory;
	@Autowired
	private GameFactory gameFactory;
	@Autowired
	private Locale locale;
	@Autowired
	private MessageSource messageSource;
	@Autowired
	private GameOptionsService gameOptions;
	@Autowired
	private Date gameDate;
	@Autowired
	private IFontLoader fontLoader;
	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;

	private IGame game;
	// Need to inject the font as for custom controls it is not stylable
	private Font font24;
	private SceneChangeable sceneChangeable;
	private OpenPatricianSpinner lodableGames;

	public LoadGameScene() {
		super(new StackPane());
		getRoot().getStylesheets().add(this.getClass().getResource(getStyleSheetFilename()).toExternalForm());
	}


	public SceneChangeable getSceneChangeable() {
		return sceneChangeable;
	}

	public void setSceneChangeable(SceneChangeable sceneChangeable) {
		this.sceneChangeable = sceneChangeable;
	}
	protected String getStyleSheetFilename() {
		return "newGame.css";
	}
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }
    @PostConstruct
	private void createControls() {
		game = gameFactory.getGame();
		font24 = fontLoader.createDefaultFont(24);
		StackPane root = (StackPane) getRoot();

		GridPane grid = new GridPane();
		grid.getStyleClass().add("grid");
		grid.setAlignment(Pos.CENTER);
		grid.getColumnConstraints().add(new ColumnConstraints(200)); // column 1
		grid.getColumnConstraints().add(new ColumnConstraints(300)); // column 2
		grid.getColumnConstraints().add(new ColumnConstraints(250)); // column 3
		grid.getColumnConstraints().add(new ColumnConstraints(100)); // column 4

		// Load
		Label loadLbl = new Label(messageSource.getMessage("ch.sahits.game.graphic.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()));
		grid.add(loadLbl, 0, 0);

		lodableGames = new OpenPatricianSpinner(font24);
		grid.add(lodableGames, 1, 0);
		GridPane.setHalignment(lodableGames, HPos.LEFT);
		lodableGames.setOptionsMap(gameOptions.getSavegames());
		lodableGames.setMaxWidth(250);
		OpenPatricianStoneButton loadBtn = new OpenPatricianStoneButton(messageSource.getMessage("ch.sahits.game.graphic.display.scene.InGameOptionsScene.load", new Object[]{}, locale.getCurrentLocal()), font24);
		grid.add(loadBtn, 3, 0);
		GridPane.setHalignment(loadBtn, HPos.LEFT);
		loadBtn.setOnAction(createLoadAction());




		root.getChildren().add(grid);

		clientServerEventBus.register(this);
	}

	@Subscribe
	public void handleGameStateChange(GameStateChange stateChange) {
		if (stateChange.getStatusChange() == EGameStatusChange.GAME_INITIALISATION_COMPLETE) {
			logger.info("Load and replace the game");
			  gameOptions.load((String) lodableGames.getSelectedValue());
		}
		if (stateChange.getStatusChange() == EGameStatusChange.GAME_LOADED) {

			uiFactory.invalidate();

			final MainGameScene mainGameScene1 = uiFactory.getMainGameScene(getRoot().getWidth(), getRoot().getHeight());
			mainGameScene1.initializeGameView(null);
            Platform.runLater(() ->
                changeScene(mainGameScene1)
            );
		}
	}


	private EventHandler<MouseEvent> createLoadAction() {
		return event -> {
			NewGame newGameDTO = NewGame.builder()
					.difficulty(Difficulty.CHANDLER)
					.firstName("FirstName")
					.lastName("Lastname")
					.hometown("BREMEN")
					.objective(EObjective.ENDLESS)
					.speed(EGameSpeed.SLOW)
					.startYear(1300)
					.male(true)
					.mapName("/standartHanseaticMap.xml")
					.environmentInitialisation(true) // environment is not yet initialized
					.build();
			// notify server
			// This will instanciate the UI, so it can be replaced properly
			gameDate.setStartYear(Date.DEFAULT_INVALID_START_YEAR); // will be changed
			final double width = getRoot().getWidth();
			final double height = getRoot().getHeight();
			logger.debug("Change into main game scene with dimensions: {}x{}", width, height);
			final MainGameScene mainGameScene = uiFactory.getMainGameScene(width, height);
			mainGameScene.initializeGameView(null); // force initialisation of MainGameView

			GameStateChange stateChange = new GameStateChange(EGameStatusChange.LOAD_GAME_INIT);
			stateChange.setStateChangeData(newGameDTO);
			clientServerEventBus.post(stateChange);

			// Request the view change
			while (gameDate.getStartYear() == null) {
				try {
					Thread.sleep(1);
				} catch (InterruptedException e) {
					logger.warn("Interrupted while waiting for the startyear to be set.");
				}
			}


		};
	}

	public void changeScene(OpenPatricianScene scene) {  // fixme: andi 7/19/15: code duplication
		double oldWidth = 0;
		double oldHeight = 0;
		double newWidth = sceneChangeable.getSceneWidth();
		double newHeight = sceneChangeable.getSceneHeight();
		sceneChangeable.changeScene(scene);
		// Fix needed to avoid bug https://javafx-jira.kenai.com/browse/RT-30392
		scene.widthChange(oldWidth, newWidth); // old value is wrong as that is the same as the new one
		scene.heightChange(oldHeight, newHeight);
		if (scene instanceof BaseStartupScene) {
			((BaseStartupScene)scene).setSceneChangeable(sceneChangeable); // propagate
		}
	}


}
